<?php
/**
 * User Registration process.
 *
 * @since   1.0.0
 * @package EverestForms_User_Registration
 */

defined( 'ABSPATH' ) || exit;

/**
 * EVF User Registration process class.
 */
class EVF_User_Registration_Process {

	/**
	 * Errors.
	 *
	 * @var array
	 */
	public $errors = '';

	/**
	 * Success.
	 *
	 * @var string
	 */
	public $success = '';

	/**
	 * Primary class constructor.
	 */
	public function __construct() {
		add_action( 'everest_forms_process', array( $this, 'process_user_registration' ), 30, 3 );
		add_filter( 'wp_authenticate_user', array( $this, 'check_status_on_login' ) );
		add_action( 'wp_loaded', array( $this, 'process_login' ) );
		add_filter( 'everest_forms_frontend_load', array( $this, 'display_form' ), 10, 2 );
		add_filter( 'everest_forms_process_after_filter', array( $this, 'password_value_remove' ), 10, 3 );
	}

	/**
	 * On form display actions.
	 *
	 * @since 1.0.0
	 *
	 * @param bool  $load_form Indicates whether a form should be loaded.
	 * @param array $form_data Form information.
	 *
	 * @return bool
	 */
	public function display_form( $load_form, $form_data ) {
		// If the form has user registration enabled and current Whether the current user is logged in and doesn't have a administrator capability do not proceed.
		if ( isset( $form_data['settings']['enable_user_registration'] ) && '1' === $form_data['settings']['enable_user_registration'] && is_user_logged_in() && ! current_user_can( 'administrator' ) ) {
			echo '<p class="everest-forms-notice everest-forms-notice--error">' . esc_html__( 'You are already logged in. Please logout to register New User.', 'everest-forms-user-registration' ) . ' <a href="' . esc_url( wp_logout_url( get_permalink() ) ) . '">' . esc_html__( 'Logout', 'everest-forms-user-registration' ) . '</a></p>';
			return false;
		}

		return $load_form;
	}

	/**
	 * Validate and process registration form.
	 *
	 * @param array $fields Submitted fields.
	 * @param array $entry  Submitted data.
	 * @param array $form_data Form data.
	 */
	public function process_user_registration( $fields, $entry, $form_data ) {
		if ( ! isset( $form_data['settings']['enable_user_registration'] ) || false === $form_data['settings']['enable_user_registration'] ) {
			return;
		}

		// Return if form has errors.
		$errors = evf()->task->errors;
		if ( ! empty( $errors[ $form_data['id'] ] ) ) {
			return;
		}

		$register_fields = array();
		$db_key          = array( 'user_login', 'user_email', 'display_name', 'user_url', 'first_name', 'last_name', 'description' );

		foreach ( $db_key as $value ) {
			if ( ! empty( $form_data['settings'][ 'user_registration_' . $value ] ) ) {
				$field_id                               = $form_data['settings'][ 'user_registration_' . $value ];
				$register_fields['user_data'][ $value ] = $entry['form_fields'][ $field_id ];
			}
		}

		if ( is_array( $register_fields['user_data']['user_email'] ) ) {
			$register_fields['user_data']['user_email'] = $register_fields['user_data']['user_email']['primary'];
		}

		if ( ! empty( $form_data['settings']['user_registration_role'] ) ) {
			$register_fields['user_data']['role'] = $form_data['settings']['user_registration_role'];
		}

		if ( ! empty( $form_data['settings']['user_registration_user_pass'] ) ) {
			$field_id                                  = $form_data['settings']['user_registration_user_pass'];
			$register_fields['user_data']['user_pass'] = isset( $entry['form_fields'][ $field_id ]['primary'] ) ? $entry['form_fields'][ $field_id ]['primary'] : $entry['form_fields'][ $field_id ];
		} else {
			$register_fields['user_data']['user_pass'] = wp_generate_password( 10 );
		}

		if ( ! empty( $register_fields['user_data']['user_login'] ) ) {
			$register_fields['user_data']['user_login'] = $this->check_username( $register_fields['user_data']['user_login'] );
			if ( ! $register_fields['user_data']['user_login'] ) {
				evf()->task->errors[ $form_data['id'] ]['header'] = __( 'Invalid Username', 'everest-forms-user-registration' );
				return;
			}
		} else {
			$part_of_email                              = explode( '@', $register_fields['user_data']['user_email'] );
			$register_fields['user_data']['user_login'] = $this->check_username( $part_of_email[0] );
		}

		$register_fields = apply_filters( 'everest_forms_user_registration_before_user_register_procss', $register_fields, $entry, $form_data );
		$user_id         = wp_insert_user( $register_fields['user_data'] );

		// Custom user meta.
		if ( ! empty( $form_data['settings']['registration_meta'] ) ) {
			foreach ( $form_data['settings']['registration_meta'] as $meta_key => $meta_field ) {
				if ( ! empty( $fields[ $meta_field ]['value'] ) ) {
					update_user_meta( $user_id, $meta_key, $fields[ $meta_field ]['value'] );
				}
			}
		}

		// On success.
		if ( ! is_wp_error( $user_id ) ) {
			update_user_meta( $user_id, 'evf-form-id', $form_data['id'] );
			if ( empty( $form_data['settings']['user_registration_login_options'] ) ) {
				return;
			}

			do_action( 'everest_forms_user_registration_new_user_email', $user_id, $form_data, $register_fields['user_data']['user_pass'] );

			if ( 'admin_approval' === $form_data['settings']['user_registration_login_options'] ) {
				update_user_meta( $user_id, 'evf-user-status', 0 );

				$this->success = __( 'User registered! Wait until admin approves your registration.', 'everest-forms-user-registration' );
				add_action( 'everest_forms_after_success_message', array( $this, 'show_success_message' ), 10, 2 );
				return;
			} elseif ( 'email_confirmation' === $form_data['settings']['user_registration_login_options'] ) {
				update_user_meta( $user_id, 'evf-user-status', 0 );
				$token = $this->get_token();
				update_user_meta( $user_id, 'evf_confirm_email_token', $token );
				do_action( 'everest_forms_user_registration_confirmation_email', $user_id, $form_data, $token, $register_fields['user_data']['user_pass'] );

				$this->success = __( 'User registered! Verify your email by clicking on the link sent to your email.', 'everest-forms-user-registration' );
				add_action( 'everest_forms_after_success_message', array( $this, 'show_success_message' ), 10, 2 );
				return;
			} else {
				wp_clear_auth_cookie();
				wp_set_auth_cookie( $user_id );
			}

				$redirect_url = apply_filters( 'everest_forms_user_registration_signup_redirect_url', esc_url( $form_data['settings']['redirect_to'] ), $user_id );
				wp_safe_redirect( $redirect_url );
		} else {
			evf()->task->errors[ $form_data['id'] ]['header'] = $user_id->get_error_message();
		}
	}

	/**
	 * Show sucess message after form submission.
	 *
	 * @param array $form_data Form data.
	 * @param array $entry Entry data.
	 */
	public function show_success_message( $form_data, $entry ) {
		if ( ! empty( $this->success ) ) {
			evf_clear_notices();
			evf_add_notice( $this->success, 'success' );
		}
	}

	/**
	 * Generate email token.
	 *
	 * @return string   Token.
	 */
	public function get_token() {

		$length         = 50;
		$token          = '';
		$code_alphabet  = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
		$code_alphabet .= 'abcdefghijklmnopqrstuvwxyz';
		$code_alphabet .= '0123456789';
		$max            = strlen( $code_alphabet );

		for ( $i = 0; $i < $length; $i++ ) {
			$token .= $code_alphabet[ random_int( 0, $max - 1 ) ];
		}

		return $token;
	}

	/**
	 * Login Process.
	 */
	public function process_login() {
		if ( ! isset( $_POST['evf-user-login'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( wp_unslash( $_POST['evf-nonce'] ), 'evf-user-login' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
			evf_add_notice( esc_html__( 'Nonce Error! Please reload.', 'everest-forms-user-registration' ), 'error' );
			return;
		}
		$user_login       = isset( $_POST['user_login'] ) ? evf_clean( $_POST['user_login'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$user_password    = isset( $_POST['user_password'] ) ? evf_clean( $_POST['user_password'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$remember         = ( isset( $_POST['rememberme'] ) && '1' === $_POST['rememberme'] ) ? true : false;
		$enable_recaptcha = isset( $_POST['evf-login-recaptcha'] ) ? $_POST['evf-login-recaptcha'] : false; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! ( $user_login && validate_username( $user_login ) ) ) {
			evf_add_notice( esc_html__( 'Invalid Username', 'everest-forms-user-registration' ), 'error' );
			return;
		}

		if ( ! $user_password ) {
			evf_add_notice( esc_html__( 'Password is required', 'everest-forms-user-registration' ), 'error' );
			return;
		}

		// reCAPTCHA check.
		$recaptcha_type = get_option( 'everest_forms_recaptcha_type', 'v2' );

		if ( 'v2' === $recaptcha_type ) {
			$site_key   = get_option( 'everest_forms_recaptcha_v2_site_key' );
			$secret_key = get_option( 'everest_forms_recaptcha_v2_secret_key' );
		} else {
			$site_key   = get_option( 'everest_forms_recaptcha_v3_site_key' );
			$secret_key = get_option( 'everest_forms_recaptcha_v3_secret_key' );
		}

		if ( ! empty( $site_key ) && ! empty( $secret_key ) && $enable_recaptcha ) {
			if ( ( 'v2' === $recaptcha_type && ! empty( $_POST['g-recaptcha-response'] ) ) || ( 'v3' === $recaptcha_type && ! empty( $_POST['g-recaptcha-hidden'] ) ) ) {
				$response = 'v2' === $recaptcha_type ? evf_clean( wp_unslash( $_POST['g-recaptcha-response'] ) ) : evf_clean( wp_unslash( $_POST['g-recaptcha-hidden'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$raw_data = wp_safe_remote_get( 'https://www.google.com/recaptcha/api/siteverify?secret=' . $secret_key . '&response=' . $response );

				if ( ! is_wp_error( $raw_data ) ) {
					$data = json_decode( wp_remote_retrieve_body( $raw_data ) );

					// Check reCAPTCHA response.
					if ( empty( $data->success ) || ( isset( $data->hostname ) && evf_clean( wp_unslash( $_SERVER['HTTP_HOST'] ) ) !== $data->hostname ) || ( isset( $data->action, $data->score ) && ( 'everest_form' !== $data->action && 0.5 > floatval( $data->score ) ) ) ) {// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
						evf_add_notice( esc_html__( 'Incorrect reCAPTCHA, please try again.', 'everest-forms-user-registration' ), 'error' );
						return;
					}
				}
			} else {
				evf_add_notice( esc_html__( 'reCAPTCHA is required.', 'everest-forms-user-registration' ), 'error' );
				return;
			}
		}

		$user_data = array(
			'user_login'    => $user_login,
			'user_password' => $user_password,
			'remember'      => $remember,
		);

		$user = wp_signon( $user_data, is_ssl() );

		if ( is_wp_error( $user ) ) {
			evf_add_notice( $user->get_error_message(), 'error' );
			return;
		}

		$redirect_url = apply_filters( 'everest_forms_user_registration_login_redirect_url', esc_url( $_POST['evf-redirect-url'] ), $user->ID ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Check the status of an user on login
	 *
	 * @param WP_User $user User details.
	 *
	 * @return \WP_Error
	 * @throws \Exception Login status failures.
	 */
	public function check_status_on_login( WP_User $user ) {

		$status      = get_user_meta( $user->ID, 'evf-user-status', true );
		$email_token = get_user_meta( $user->ID, 'evf_confirm_email_token', true );
		$interaction = null;

		do_action( 'everest_forms_user_registration_user_before_check_status_on_login', $status, $user );

		switch ( $status ) {
			case '0':
				if ( $email_token ) {
					$message = '<strong>' . __( 'ERROR:', 'everest-forms-user-registration' ) . '</strong> ' . __( 'Please check your email to verify your account.', 'everest-forms-user-registration' );
				} else {
					$message = '<strong>' . __( 'ERROR:', 'everest-forms-user-registration' ) . '</strong> ' . __( 'Your account is still pending approval.', 'everest-forms-user-registration' );
				}

				$interaction = new WP_Error( 'pending_approval', $message );
				break;
			case '-1':
				$message = '<strong>' . __( 'ERROR:', 'everest-forms-user-registration' ) . '</strong> ' . __( 'Your account has been denied.', 'everest-forms-user-registration' );

				$interaction = new WP_Error( 'denied_access', $message );
				break;
			default:
				$interaction = $user;
				break;
		}

		return $interaction;
	}

	/**
	 * Check if username already exists in case of optional username
	 * And while stripping through email address and incremet last number by 1.
	 *
	 * @param  string $username Username.
	 * @return string
	 */
	public function check_username( $username ) {

		if ( ! validate_username( $username ) ) {
			return false;
		}
		if ( username_exists( $username ) ) {
			$last_char = substr( $username, -1 );

			if ( is_numeric( $last_char ) ) {
				$strip_last_char = substr( $username, 0, -1 );
				$last_char++;
				$username = $strip_last_char . $last_char;
				$username = $this->check_username( $username );

				return $username;
			} else {
				$username = $username . '_1';
				$username = $this->check_username( $username );

				return $username;
			}
		}

		return $username;
	}

	/**
	 * Real password with dummy one.
	 *
	 * @param array $fields Form fields.
	 * @param array $entry Entry data.
	 * @param array $form_data Form data.
	 */
	public function password_value_remove( $fields, $entry, $form_data ) {

		if ( ! isset( $form_data['settings']['enable_user_registration'] ) || '1' !== $form_data['settings']['enable_user_registration'] ) {
			return $fields;
		}

		foreach ( $fields as $id => $field ) {
			if ( 'password' === $field['type'] ) {
				$fields[ $id ]['value'] = '**********';
			}
		}

		return $fields;
	}
}

new EVF_User_Registration_Process();
