<?php
/**
 * EverestForms Style Customizer setup
 *
 * @package EverestForms_Style_Customizer
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main EverestForms Style Customizer Class.
 *
 * @class EverestForms_Style_Customizer
 */
final class EverestForms_Style_Customizer {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		// Load plugin text domain.
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		// Checks with Everest Forms Pro is installed.
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.7.2', '>=' ) ) {
			// Checks with Everest Forms Pro is installed.
			if ( defined( 'EFP_VERSION' ) && version_compare( EFP_VERSION, '1.3.6', '>=' ) ) {
				$this->configs();
				$this->includes();

				// Hooks.
				add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
				add_action( 'everest_forms_shortcode_scripts', array( $this, 'enqueue_shortcode_scripts' ) );
				add_action( 'everest_forms_builder_content_fields', array( $this, 'output_form_designer' ) );
				add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );
			} else {
				add_action( 'admin_notices', array( $this, 'everest_forms_pro_missing_notice' ) );
			}
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_missing_notice' ) );
		}
	}

	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		// If the single instance hasn't been set, set it now.
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-style-customizer/everest-forms-style-customizer-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-style-customizer-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-style-customizer' );

		load_textdomain( 'everest-forms-style-customizer', WP_LANG_DIR . '/everest-forms-style-customizer/everest-forms-style-customizer-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-style-customizer', false, plugin_basename( dirname( EVF_STYLE_CUSTOMIZER_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Configs.
	 */
	private function configs() {
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-form-wrapper-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-field-label-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-field-sublabel-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-field-description-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-field-styles-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-file-upload-styles-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-radio-checkbox-styles-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-section-title-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-button-configs.php';
		require_once dirname( __FILE__ ) . '/configs/evf-style-customizer-submission-message-configs.php';
	}

	/**
	 * Includes.
	 */
	private function includes() {
		require_once dirname( __FILE__ ) . '/class-evf-style-customizer-api.php';
	}

	/**
	 * Get the customizer url.
	 */
	private function get_customizer_url() {
		$form_id        = isset( $_GET['form_id'] ) ? absint( $_GET['form_id'] ) : false; // phpcs:ignore WordPress.Security.NonceVerification
		$customizer_url = esc_url_raw(
			add_query_arg(
				array(
					'evf-style-customizer' => true,
					'form_id'              => $form_id,
					'return'               => rawurlencode(
						add_query_arg(
							array(
								'page'    => 'evf-builder',
								'tab'     => 'fields',
								'form_id' => $form_id,
							),
							admin_url( 'admin.php' )
						)
					),
				),
				admin_url( 'customize.php' )
			)
		);

		return $customizer_url;
	}

	/**
	 * Plugin Updater.
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return EVF_Plugin_Updater::updates( EVF_STYLE_CUSTOMIZER_PLUGIN_FILE, 16166, EVF_STYLE_CUSTOMIZER_VERSION );
		}
	}

	/**
	 * Enqueue scripts.
	 */
	public function admin_enqueue_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';

		// Register admin scripts.
		wp_register_style( 'everest-forms-customize-admin', plugins_url( '/assets/css/customize-admin.css', EVF_STYLE_CUSTOMIZER_PLUGIN_FILE ), array(), EVF_STYLE_CUSTOMIZER_VERSION );

		// Add RTL support for admin styles.
		wp_style_add_data( 'everest-forms-customize-admin', 'rtl', 'replace' );

		// Admin styles for EVF pages only.
		if ( in_array( $screen_id, evf_get_screen_ids(), true ) ) {
			wp_enqueue_style( 'everest-forms-customize-admin' );
		}
	}

	/**
	 * Enqueue shortcode scripts.
	 *
	 * @param array $atts Shortcode Attributes.
	 */
	public function enqueue_shortcode_scripts( $atts ) {
		$form_id       = absint( $atts['id'] );
		$upload_dir    = wp_upload_dir( null, false );
		$style_options = get_option( 'everest_forms_styles' );

		// Enqueue shortcode styles.
		if ( file_exists( trailingslashit( $upload_dir['basedir'] ) . 'everest_forms_styles/everest-forms-' . $form_id . '.css' ) ) {
			wp_enqueue_style( 'everest-forms-style-' . $form_id, esc_url_raw( set_url_scheme( $upload_dir['baseurl'] . '/everest_forms_styles/everest-forms-' . $form_id . '.css' ) ), array(), filemtime( trailingslashit( $upload_dir['basedir'] ) . 'everest_forms_styles/everest-forms-' . $form_id . '.css' ), 'all' );
		}

		// Enqueue google fonts styles.
		if ( isset( $style_options[ $form_id ]['wrapper']['font_family'] ) && '' !== $style_options[ $form_id ]['wrapper']['font_family'] ) {
			wp_enqueue_style( 'everest-forms-google-fonts', 'https://fonts.googleapis.com/css?family=' . evf_clean( $style_options[ $form_id ]['wrapper']['font_family'] ), array(), EVF_STYLE_CUSTOMIZER_VERSION, 'all' );
		}
	}

	/**
	 * Output form designer.
	 */
	public function output_form_designer() {
		?>
		<a href="<?php echo esc_url( $this->get_customizer_url() ); ?>" class="everest-forms-designer-icon tips" title="<?php esc_attr_e( 'Form Designer', 'everest-forms-style-customizer' ); ?>">
			<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M16.85 9.08l-4.07 4.06a.57.57 0 0 0-.06.81l.06.05 1.37 1.37.1.1c.09-.09.16-.18.25-.25a.78.78 0 0 1 1.06 0q1.55 1.53 3.07 3.07a.79.79 0 0 1 0 1.11q-.58.59-1.17 1.17a.8.8 0 0 1-1.13 0l-3-3-.1-.12a.8.8 0 0 1 .09-1l.19-.19-.1-.1-1.76-1.76a1.14 1.14 0 0 1 0-1.61l4.07-4a.63.63 0 0 0 0-.89c-.48-.44-.94-.93-1.41-1.4a.57.57 0 0 0-.81 0l-.33.31.07.08.54.54a.71.71 0 0 1 0 1l-6.58 6.5a.71.71 0 0 1-.81.2.78.78 0 0 1-.27-.13l-2-2a.71.71 0 0 1 0-1l6.51-6.51a.72.72 0 0 1 1-.09l.09.09.49.49.15.02.7-.71a1.14 1.14 0 0 1 1.61 0l2.17 2.17a1.14 1.14 0 0 1 0 1.61z"/></svg>
		</a>
		<?php
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EVF_STYLE_CUSTOMIZER_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-add-ons/style-customizer/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms Style Customizer documentation', 'everest-forms-style-customizer' ) . '">' . esc_html__( 'Docs', 'everest-forms-style-customizer' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}

	/**
	 * Everest Forms fallback notice.
	 */
	public function everest_forms_missing_notice() {
		/* translators: %s: everest-forms version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Style Customizer requires at least %s or later to work!', 'everest-forms-style-customizer' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms 1.7.1', 'everest-forms-style-customizer' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Everest Forms Pro fallback notice.
	 */
	public function everest_forms_pro_missing_notice() {
		/* translators: %s: everest-forms-pro version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Style Customizer depends on the last version of %s or later to work!', 'everest-forms-style-customizer' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms Pro 1.3.6', 'everest-forms-style-customizer' ) . '</a>' ) . '</p></div>';
	}
}
