<?php
/**
 * EverestForms User Registration setup
 *
 * @package EverestForms_User_Registration
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main EverestForms User Registration Class.
 *
 * @class EverestForms_User_Registration
 */
final class EverestForms_User_Registration {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		// Load plugin text domain.
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
		add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );

		// Checks with Everest Forms is installed.
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.5.0', '>=' ) ) {

			// Checks with Everest Forms Pro is installed.
			if ( defined( 'EFP_VERSION' ) && version_compare( EFP_VERSION, '1.2.6', '>=' ) ) {
				$this->define_constants();
				$this->includes();

				// Hooks.
				add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
			} else {
				add_action( 'admin_notices', array( $this, 'everest_forms_pro_missing_notice' ) );
			}
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_missing_notice' ) );
		}
	}

	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		// If the single instance hasn't been set, set it now.
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-user-registration/everest-forms-user-registration-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-user-registration-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-user-registration' );

		load_textdomain( 'everest-forms-user-registration', WP_LANG_DIR . '/everest-forms-user-registration/everest-forms-user-registration-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-user-registration', false, plugin_basename( dirname( EVF_USER_REGISTRATION_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EVF_USER_REGISTRATION_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-add-ons/user-registration/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms User Registration documentation', 'everest-forms-user-registration' ) . '">' . esc_html__( 'Docs', 'everest-forms-user-registration' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}

	/**
	 * Define EVFUR Constants.
	 */
	private function define_constants() {
		$this->define( 'EVFUR_ABSPATH', dirname( EVF_USER_REGISTRATION_PLUGIN_FILE ) . '/' );
		$this->define( 'EVFUR_PLUGIN_BASENAME', plugin_basename( EVF_USER_REGISTRATION_PLUGIN_FILE ) );
	}

	/**
	 * Define constant if not already set.
	 *
	 * @param string      $name  Constant name.
	 * @param string|bool $value Constant value.
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Includes.
	 */
	private function includes() {
		include_once dirname( __FILE__ ) . '/admin/class-evfur-form-settings.php';
		include_once dirname( __FILE__ ) . '/class-everest-forms-user-activation.php';
		include_once dirname( __FILE__ ) . '/class-everest-forms-user-registration-process.php';
		include_once dirname( __FILE__ ) . '/class-everest-forms-login-shortcode.php';
	}

	/**
	 * Plugin Updater.
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return EVF_Plugin_Updater::updates( EVF_USER_REGISTRATION_PLUGIN_FILE, 22439, EVF_USER_REGISTRATION_VERSION );
		}
	}

	/**
	 * Enqueue scripts.
	 */
	public function admin_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		$suffix    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Register admin scripts.
		wp_register_script( 'everest-forms-user-registration-builder', plugins_url( "/assets/js/admin/admin{$suffix}.js", EVF_USER_REGISTRATION_PLUGIN_FILE ), array( 'jquery', 'jquery-ui-sortable' ), EVF_USER_REGISTRATION_VERSION, true );
		wp_localize_script(
			'everest-forms-user-registration-builder',
			'evf_user_registration_params',
			array(
				'i18n_disabled_message' => esc_html__( 'User registration is currently disabled. Please enable it to register users with Everest Forms.', 'everest-forms-user-registration' ),
			)
		);

		// Admin scripts for EVF builder page.
		if ( 'everest-forms_page_evf-builder' === $screen_id ) {
			wp_enqueue_script( 'everest-forms-user-registration-builder' );
		}
	}

	/**
	 * Everest Forms fallback notice.
	 */
	public function everest_forms_missing_notice() {
		/* translators: %s: everest-forms version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - User Registration requires at least %s or later to work!', 'everest-forms-user-registration' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms 1.5.0', 'everest-forms-user-registration' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Everest Forms Pro fallback notice.
	 */
	public function everest_forms_pro_missing_notice() {
		/* translators: %s: everest-forms-pro version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - User Registration depends on the last version of %s or later to work!', 'everest-forms-user-registration' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms Pro 1.2.6', 'everest-forms-user-registration' ) . '</a>' ) . '</p></div>';
	}
}
