<?php
/**
 * Everest Forms Login shortcode
 *
 * @since   1.0.0
 * @package EverestFormsUserRegistration\Admin
 */

defined( 'ABSPATH' ) || exit;

/**
 * Everest Forms Login Shortcode class.
 */
class Everest_Forms_Login_Shortcode {

	/**
	 * Primary class constructor.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'load_shortcode' ) );
		add_filter( 'evf_get_template', array( $this, 'template_location' ), 10, 2 );
	}

	/**
	 * Load shortcode.
	 */
	public function load_shortcode() {
		add_shortcode( apply_filters( 'everest_forms_user_login_shortcode_tag', 'everest_forms_user_login' ), array( $this, 'user_login' ) );
	}

	/**
	 * User Login.
	 *
	 * @param array $attr Login attributes.
	 */
	public function user_login( $attr ) {
		ob_start();

		if ( is_user_logged_in() ) {
			echo '<p class="everest-forms-notice everest-forms-notice--error">' . esc_html__( 'You are already logged in. Please logout to register New User.', 'everest-forms-user-registration' ) . ' <a href="' . esc_url_raw( wp_logout_url( get_permalink() ) ) . '">' . esc_html__( 'Logout', 'everest-forms-user-registration' ) . '</a></p>';
		} else {

			$nonce        = wp_create_nonce( 'evf-user-login' );
			$redirect_url = isset( $attr['redirect_url'] ) ? $attr['redirect_url'] : get_home_url();
			$recaptcha    = $this->get_recaptcha_node( isset( $attr['recaptcha'] ) ? $attr['recaptcha'] : false );

			evf_get_template(
				'user-login.php',
				array(
					'nonce'            => $nonce,
					'redirect_url'     => $redirect_url,
					'recaptcha_status' => isset( $attr['recaptcha'] ) ? $attr['recaptcha'] : false,
					'recaptcha_node'   => $recaptcha,
				)
			);
		}
		$output = ob_get_clean();
		return $output;
	}


	/**
	 * Change Template Location.
	 *
	 * @param mixed $located path to template.
	 * @param mixed $template_name name of template.
	 */
	public function template_location( $located, $template_name ) {
		if ( 'user-login.php' === $template_name ) {
			$located = EVFUR_ABSPATH . 'templates/user-login.php';
		}
		return $located;
	}

	/**
	 * Recaptcha for login form.
	 *
	 * @param bool $enable_recaptcha recaptcha status.
	 */
	public function get_recaptcha_node( $enable_recaptcha ) {
		$recaptcha_node = '';

		if ( ! $enable_recaptcha ) {
			return $recaptcha_node;
		}
		$recaptcha_type   = get_option( 'everest_forms_recaptcha_type', 'v2' );
		$enable_recaptcha = true;

		if ( 'v2' === $recaptcha_type ) {
			$site_key            = get_option( 'everest_forms_recaptcha_v2_site_key' );
			$secret_key          = get_option( 'everest_forms_recaptcha_v2_secret_key' );
			$invisible_recaptcha = get_option( 'everest_forms_recaptcha_v2_invisible', 'no' );
		} else {
			$site_key   = get_option( 'everest_forms_recaptcha_v3_site_key' );
			$secret_key = get_option( 'everest_forms_recaptcha_v3_secret_key' );
		}

		if ( ! $site_key || ! $secret_key ) {
			return '';
		}

		if ( $enable_recaptcha ) {
			$data = array(
				'sitekey' => trim( sanitize_text_field( $site_key ) ),
			);

			// Load reCAPTCHA support if form supports it.
			if ( $site_key && $secret_key ) {
				if ( 'v2' === $recaptcha_type ) {
					$recaptcha_api = apply_filters( 'everest_forms_frontend_recaptcha_url', 'https://www.google.com/recaptcha/api.js?onload=EVFRecaptchaLoad&render=explicit' );
					if ( 'yes' === $invisible_recaptcha ) {
						$recaptcha_inline = 'var EVFRecaptchaLoad = function(){jQuery(".g-recaptcha").each(function(index, el){var recaptchaID = grecaptcha.render(el,{},true); grecaptcha.execute(recaptchaID);});};';
					} else {
						$recaptcha_inline  = 'var EVFRecaptchaLoad = function(){jQuery(".g-recaptcha").each(function(index, el){grecaptcha.render(el,{callback:function(){EVFRecaptchaCallback(el);}},true);});};';
						$recaptcha_inline .= 'var EVFRecaptchaCallback = function(el){jQuery(el).parent().find(".evf-recaptcha-hidden").val("1").valid();};';
					}
				} else {
					$recaptcha_api    = apply_filters( 'everest_forms_frontend_recaptcha_url', 'https://www.google.com/recaptcha/api.js?render=' . $site_key );
					$recaptcha_inline = 'grecaptcha.ready( function() { grecaptcha.execute( "' . $site_key . '", { action: "everest_form" } ).then( function( token ) { jQuery( ".evf-recaptcha-hidden" ).val( token ); } ) } )';
				}

				// Enqueue reCaptcha scripts.
				wp_enqueue_script( 'evf-recaptcha', $recaptcha_api, array( 'jquery' ), '2.0.0', false );

				// Load reCaptcha callback once.
				static $count = 1;
				if ( 1 === $count ) {
					wp_add_inline_script( 'evf-recaptcha', $recaptcha_inline );
					$count++;
				}

				if ( 'v2' === $recaptcha_type && 'yes' === $invisible_recaptcha ) {
					// Output the reCAPTCHA container.
					$data['size']    = 'invisible';
					$recaptcha_node .= '<div class="evf-recaptcha-container recaptcha-hidden" >';
					$recaptcha_node .= '<div ' . evf_html_attributes( '', array( 'g-recaptcha' ), $data ) . '></div>';
					$recaptcha_node .= '</div>';
				} else {
					// Output the reCAPTCHA container.
					$class           = 'v3' === $recaptcha_type ? 'recaptcha-hidden' : '';
					$recaptcha_node .= '<div class="evf-recaptcha-container ' . $class . '" >';
					$recaptcha_node .= '<div ' . evf_html_attributes( '', array( 'g-recaptcha' ), $data ) . '></div>';
					$recaptcha_node .= '<input type="text" name="g-recaptcha-hidden" class="evf-recaptcha-hidden" style="position:absolute!important;clip:rect(0,0,0,0)!important;height:1px!important;width:1px!important;border:0!important;overflow:hidden!important;padding:0!important;margin:0!important;" required>';
					$recaptcha_node .= '</div>';
				}
			}
		}

		return $recaptcha_node;
	}
}
new Everest_Forms_Login_Shortcode();
