<?php
/**
 * Main plugin class.
 *
 * @package EverestForms\SaveAndContinue
 *
 * @since 1.0.0
 */

namespace EverestForms\SaveAndContinue;

/**
 * Save and Continue Feature Class.
 */
class SaveAndContinue {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'everest_forms_display_submit_after', array( $this, 'display_save_continue_button' ) );

		// AJAX events.
		add_action( 'wp_ajax_nopriv_everest_forms_sc_email_send', array( $this, 'save_and_continue_email_send' ) );
		add_action( 'wp_ajax_everest_forms_sc_email_send', array( $this, 'save_and_continue_email_send' ) );

		// Corn Job.
		add_action( 'everest_forms_cleanup_expired_entries', array( $this, 'cleanup_expired_entries' ) );

		// Remove Form Data.
		add_action( 'everest_forms_after_success_message', array( $this, 'after_success_message' ) );

		// Smart Tags.
		add_filter( 'everest_forms_process_smart_tags', array( $this, 'process_save_and_continue_smart_tag' ), 10, 4 );
		add_filter( 'everest_forms_frontend_form_atts', array( $this, 'process_save_and_continue_atts' ), 10, 2 );
		add_filter( 'everest_forms_smart_tags', array( $this, 'save_and_continue_smart_tags' ) );
	}

	/**
	 * Display Save and Continue Button.
	 *
	 * @param mixed $form_data Form data.
	 *
	 * @since 1.0.0
	 */
	public function display_save_continue_button( $form_data ) {
		// Check if Save and Continue Enabled.
		if ( ! array_key_exists( 'enable_save_and_continue', $form_data['settings'] ) || '1' !== $form_data['settings']['enable_save_and_continue'] ) {
			return;
		}
		$form_id    = absint( $form_data['id'] );
		$settings   = isset( $form_data['settings'] ) ? $form_data['settings'] : array();
		$submit     = apply_filters( 'everest_forms_field_submit', isset( $settings['save_and_continue_submit'] ) ? $settings['save_and_continue_submit'] : __( 'Save and Continue', 'everest-forms-save-and-continue' ), $form_data );
		$submit_btn = evf_string_translation( $form_data['id'], 'save_and_continue_button', $submit );
		$process    = '';
		$classes    = isset( $form_data['settings']['save_and_continue_submit_class'] ) ? evf_sanitize_classes( $form_data['settings']['save_and_continue_submit_class'] ) : '';
		$parts      = ! empty( self::$parts[ $form_id ] ) ? self::$parts[ $form_id ] : array();
		$visible    = ! empty( $parts ) ? 'style="display:none"' : '';

		// Check for submit button processing-text.
		if ( ! isset( $settings['submit_button_processing_text'] ) ) {
			$process = 'data-process-text="' . esc_attr__( 'Processing&hellip;', 'everest-forms-save-and-continue' ) . '"';
		} elseif ( ! empty( $settings['submit_button_processing_text'] ) ) {
			$process = 'data-process-text="' . esc_attr( evf_string_translation( $form_data['id'], 'processing_text', $settings['submit_button_processing_text'] ) ) . '"';
		}

		// Submit button area.
		$conditional_id = 'evf-submit-' . $form_id;
		if ( isset( $form_data['settings']['submit']['connection_1']['conditional_logic_status'] ) && '1' === $form_data['settings']['submit']['connection_1']['conditional_logic_status'] ) {
			$con_rules = array(
				'conditional_option' => isset( $form_data['settings']['submit']['connection_1']['conditional_option'] ) ? $form_data['settings']['submit']['connection_1']['conditional_option'] : '',
				'conditionals'       => isset( $form_data['settings']['submit']['connection_1']['conditionals'] ) ? $form_data['settings']['submit']['connection_1']['conditionals'] : '',
			);
		} else {
			$con_rules = '';
		}

		$conditional_rules = wp_json_encode( $con_rules );

		printf(
			"&nbsp;<button type='submit' name='everest_forms[save_and_continue]' class='everest-forms-save-and-continue-button button evf-save-and-continue %s' id='evf-save-and-continue-%d' value='evf-save-and-continue' %s conditional_rules='%s' conditional_id='%s' %s>%s</button>",
			esc_attr( $classes ),
			esc_attr( $form_id ),
			esc_attr( $process ),
			esc_attr( $conditional_rules ),
			esc_attr( $conditional_id ),
			esc_attr( $visible ),
			esc_html( $submit_btn )
		);
	}

	/**
	 * Send url to Email.
	 *
	 * @since 1.0.0
	 */
	public function save_and_continue_email_send() {
		// Check nonce for form submission.
		if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'everest-forms-save-and-continue-email' ) ) { // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			wp_send_json_error(
				array(
					'message' => __( 'Nonce verification failed, please try again by reloading page.', 'everest-forms-save-and-continue' ),
					'title'   => __( 'Nonce verification failed, please try again by reloading page.', 'everest-forms-save-and-continue' ),
				)
			);
		}

		if ( isset( $_POST['email'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$data = $_POST; // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		if ( ! isset( $data ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'Post data is Invalid', 'everest-forms-save-and-continue' ),
					'title'   => __( 'Post data is Invalid', 'everest-forms-save-and-continue' ),
				)
			);
		}

		$to_email = sanitize_email( $data['email'] );
		$nonce    = $data['nonce'];
		$url      = esc_url_raw( $data['url'] );
		$entry_id = $data['entry_id'];

		$entry_id         = isset( $entry_id ) ? absint( $entry_id ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$entry            = evf_get_entry( $entry_id, true );
		$form_id          = $entry->form_id;
		$form_data        = evf()->form->get( $form_id, array( 'content_only' => true ) );
		$form_data['url'] = $url;

		// Check if Save and Continue Enabled.
		if ( ! array_key_exists( 'enable_save_and_continue', $form_data['settings'] ) || '1' !== $form_data['settings']['enable_save_and_continue'] ) {
			wp_send_json_error(
				array(
					'message' => __( 'Save and Continue is disabled', 'everest-forms-save-and-continue' ),
					'title'   => __( 'Save and Continue is disabled', 'everest-forms-save-and-continue' ),
				)
			);
		}

		if ( ! array_key_exists( 'connection_save_and_continue', $form_data['settings']['email'] ) || '1' !== $form_data['settings']['email']['connection_save_and_continue']['enable_email_notification'] ) {
			wp_send_json_error(
				array(
					'message' => __( 'Save and Continue Email is disabled', 'everest-forms-save-and-continue' ),
					'title'   => __( 'Save and Continue Email is disabled', 'everest-forms-save-and-continue' ),
				)
			);
		}

		$notification = $form_data['settings']['email']['connection_save_and_continue'];

		$email        = array();
		$evf_to_email = $to_email;
		// Setup email properties.
		$email['subject']        = ! empty( $notification['evf_email_subject'] ) ? $notification['evf_email_subject'] : sprintf( esc_html__( 'Form has been saved !', 'everest-forms-save-and-continue' ) );
		$email['address']        = explode( ',', apply_filters( 'everest_forms_process_smart_tags', $evf_to_email, $form_data, evf_decode( $entry->fields ), $entry_id ) );
		$email['address']        = array_map( 'sanitize_email', $email['address'] );
		$email['sender_name']    = ! empty( $notification['evf_from_name'] ) ? $notification['evf_from_name'] : get_bloginfo( 'name' );
		$email['sender_address'] = ! empty( $notification['evf_from_email'] ) ? $notification['evf_from_email'] : get_option( 'admin_email' );
		$email['reply_to']       = ! empty( $notification['evf_reply_to'] ) ? $notification['evf_reply_to'] : $email['sender_address'];
		$email['message']        = ! empty( $notification['evf_email_message'] ) ? $notification['evf_email_message'] : '{all_fields}';
		$email                   = apply_filters( 'everest_forms_entry_email_atts', $email, $form_data['form_fields'], $entry, $form_data );

		// Create new email.
		$emails = new \EVF_Emails();
		$emails->__set( 'form_data', $form_data );
		$emails->__set( 'fields', array() );
		$emails->__set( 'entry_id', $entry_id );
		$emails->__set( 'from_name', $email['sender_name'] );
		$emails->__set( 'from_address', $email['sender_address'] );
		$emails->__set( 'reply_to', $email['reply_to'] );

		// Maybe include Cc and Bcc email addresses.
		if ( 'yes' === get_option( 'everest_forms_enable_email_copies' ) ) {

			if ( ! empty( $notification['evf_carboncopy'] ) ) {
				$emails->__set( 'cc', $notification['evf_carboncopy'] );
			}

			if ( ! empty( $notification['evf_blindcarboncopy'] ) ) {
				$emails->__set( 'bcc', $notification['evf_blindcarboncopy'] );
			}
		}

		$emails = apply_filters( 'everest_forms_entry_email_before_send', $emails );

		// Send entry email.
		foreach ( $email['address'] as $address ) {
			$emails->send( trim( $address ), $email['subject'], $email['message'] );
		}

		wp_send_json_success(
			array(
				'message' => __( 'Email has been successfully sent', 'everest-forms-save-and-continue' ),
				'title'   => __( 'Email has been successfully sent', 'everest-forms-save-and-continue' ),
			)
		);
	}

	/**
	 * Scheduled Task.
	 *
	 * @since 1.0.0
	 */
	public function cleanup_expired_entries() {

		if ( ! is_admin() ) {
			return;
		}

		$forms = evf()->form->get_multiple( array(), true );

		foreach ( $forms as $form ) {
			$time    = array(
				'week'      => ' + 7 days',
				'two_weeks' => ' + 14 days',
				'month'     => ' + 30 days',
			);
			$expires = array_key_exists( 'save_and_continue_time', $form['settings'] ) ? $time[ $form['settings']['save_and_continue_time'] ] : $time['week'];
			$entries = evf_search_entries(
				array(
					'form_id' => $form['id'],
					'status'  => 'draft',
				)
			);

			foreach ( $entries as $key => $entry ) {
				$entry = evf_get_entry( $entry );
				if ( strtotime( $entry->date_created . ' - 10 days' ) < time() ) {
					\EVF_Admin_Entries::remove_entry( $entry->entry_id );
				}
			}
		}
	}

	/**
	 * Remnove Form Data.
	 *
	 * @param mixed $form_data Form Data.
	 *
	 * @since 1.0.0
	 */
	public function after_success_message( $form_data ) {
		echo '<script>localStorage.removeItem("evf-form-' . esc_html( $form_data['id'] ) . '");</script>';
	}

	/**
	 * Save and Continue smart tags.
	 *
	 * @param mixed $tags Smart Tags.
	 *
	 * @since 1.0.0
	 */
	public function save_and_continue_smart_tags( $tags ) {
		return array_merge(
			$tags,
			array(
				'save_and_continue_url'         => esc_html__( 'Save and Continue URL', 'everest-forms-save-and-continue' ),
				'save_and_continue_email_input' => esc_html__( 'Save and Continue Email Input', 'everest-forms-save-and-continue' ),
				'save_and_continue_time'        => esc_html__( 'Save and Continue Time', 'everest-forms-save-and-continue' ),
			)
		);
	}

	/**
	 * Save and Continue Form Atts.
	 *
	 * @param mixed $atts Atts.
	 * @param mixed $form_data Form Data.
	 */
	public function process_save_and_continue_atts( $atts, $form_data ) {
		if ( ! array_key_exists( 'save_and_continue_offline_form', $form_data['settings'] ) || '1' !== $form_data['settings']['save_and_continue_offline_form'] ) {
			return $atts;
		}
		$atts['class'][] = 'everest-form-sc-offline-form';
		return $atts;
	}

	/**
	 * Installation - Adding 'token' field to entries table.
	 *
	 * @since 1.0.0
	 */
	public static function install() {
		global $wpdb;

		/**
		 * Before updating with DBDELTA, add token column to entries table schema.
		 */
		if ( $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}evf_entries';" ) ) {
			if ( ! $wpdb->get_var( "SHOW COLUMNS FROM `{$wpdb->prefix}evf_entries` LIKE 'token';" ) ) {
				$wpdb->query( "ALTER TABLE {$wpdb->prefix}evf_entries ADD `token` longtext NULL AFTER `fields`;" );
			}
		}

		wp_clear_scheduled_hook( 'everest_forms_cleanup_expired_entries' );
		wp_schedule_event( time() + ( 3 * HOUR_IN_SECONDS ), 'daily', 'everest_forms_cleanup_expired_entries' );
	}

	/**
	 * Email Input.
	 *
	 * @since 1.0.0
	 */
	public function email_input() {
		ob_start();
		$nonce = wp_create_nonce( 'everest-forms-save-and-continue-email' );
		echo '<div class="everest-forms-save-and-continue">';
		echo '<input type="hidden" class="email-input-nonce" id="email-input-nonce" value="' . esc_attr( $nonce ) . '">';
		echo '<input type="email" class="email-input" id="email-input" placeholder="mail@mail.com">';
		echo '</div>';
		$output = ob_get_clean();
		return $output;
	}

	/**
	 * Process and parse smart tags.
	 *
	 * @param string $content The string to preprocess.
	 * @param array  $form_data Array of the form data.
	 * @param array  $fields Array of the form data.
	 * @param int    $entry_id id of the form data.
	 *
	 * @since 1.0.0
	 */
	public function process_save_and_continue_smart_tag( $content, $form_data, $fields = array(), $entry_id = 0 ) {
		// Other Smart tags.
		preg_match_all( '/\{(.+?)\}/', $content, $tags );

		if ( ! empty( $tags[1] ) ) {

			foreach ( $tags[1] as $key => $tag ) {

				switch ( $tag ) {
					case 'save_and_continue_email_input':
						$email_input = $this->email_input();
						$content     = str_replace( '{' . $tag . '}', $email_input, $content );
						break;

					case 'save_and_continue_url':
						$continue_url = '<a class="sc-short-code-url" href="' . ( array_key_exists( 'url', $form_data ) ? $form_data['url'] : '' ) . '">' . ( array_key_exists( 'url', $form_data ) ? $form_data['url'] : '' ) . '</a>';
						$content      = str_replace( '{' . $tag . '}', $continue_url, $content );
						break;

					case 'save_and_continue_time':
						$period  = array(
							'week'      => '7 days',
							'two_weeks' => '14 days',
							'month'     => '30 days',
						);
						$time    = $period[ isset( $form_data['settings']['save_and_continue_time'] ) ? $form_data['settings']['save_and_continue_time'] : 'week' ];
						$content = str_replace( '{' . $tag . '}', $time, $content );
						break;
				}
			}
		}

		return $content;
	}
}
