<?php
/**
 * Main plugin class.
 *
 * @package EverestForms\SaveAndContinue
 * @since   1.0.0
 */

namespace EverestForms\SaveAndContinue;

use EverestForms\SaveAndContinue\Admin\Setting;

/**
 * Main plugin class.
 *
 * @since 1.0.0
 */
class Plugin {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 *
	 * @since 1.0.0
	 */
	protected static $instance;

	/**
	 * Cloning is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'everest-forms-save-and-continue' ), '1.0.0' );
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'everest-forms-save-and-continue' ), '1.0.0' );
	}

	/**
	 * Main plugin class instance.
	 *
	 * @since 1.0.0
	 *
	 * Ensures only one instance of the plugin is loaded or can be loaded.
	 *
	 * @return object Main instance of the class.
	 */
	final public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new static();
		}
		return static::$instance;
	}

	/**
	 * Plugin Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
		add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );

		// Checks with Everest Forms is installed.
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.7.7', '>=' ) ) {

			// Checks with Everest Forms Pro is installed.
			if ( defined( 'EFP_VERSION' ) && version_compare( EFP_VERSION, '1.4.2', '>=' ) ) {
				// Hooks.
				add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
				// Enqueing Scripts.
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
				add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );

				// Initializing Plugins Features.
				$this->init();
			} else {
				add_action( 'admin_notices', array( $this, 'everest_forms_pro_missing_notice' ) );
			}
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_missing_notice' ) );
		}

	}

	/**
	 * Frontend Enqueue scripts.
	 *
	 * @since 1.0.0
	 */
	public function frontend_enqueue_scripts() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Enqueue Scripts.
		wp_register_script( 'everest-forms-save-and-continue-script', plugins_url( "/assets/js/frontend/everest-forms-save-and-continue{$suffix}.js", EVF_SAVE_AND_CONTINUE_PLUGIN_FILE ), array(), EVF_SAVE_AND_CONTINUE_VERSION, true );
		wp_localize_script(
			'everest-forms-save-and-continue-script',
			'everest_forms_save_and_continue_script',
			array(
				'ajax_url'        => admin_url( 'admin-ajax.php', 'relative' ),
				'ajax_nonce'      => wp_create_nonce( 'ajax_sc_submission_nonce' ),
				'restore_message' => esc_html__( 'It seems you haven`t submitted your last form. Do you like to restore it?', 'everest-forms-save-and-continue' ),
				'error_essage'    => esc_html__( 'Sorry, Something unexpected happened, Please try again.', 'everest-forms-save-and-continue' ),
			)
		);
		wp_enqueue_script( 'everest-forms-save-and-continue-script' );

		wp_enqueue_style( 'everest-forms-save-and-continue-style', plugins_url( '/assets/css/everest-forms-save-and-continue.css', EVF_SAVE_AND_CONTINUE_PLUGIN_FILE ), array(), EVF_SAVE_AND_CONTINUE_VERSION );

		wp_localize_script(
			'everest-forms-save-and-continue-script',
			'everest_forms_sc_form_submission_params',
			$params = array(
				'ajax_url'            => admin_url( 'admin-ajax.php' ),
				'evf_ajax_submission' => wp_create_nonce( 'everest_forms_ajax_form_submission' ),
				'error'               => esc_html__( 'Sorry, something went wrong. Please try again', 'everest-forms-save-and-continue' ),
			)
		);
	}

	/**
	 * Admin Enqueue Scripts.
	 *
	 * @since 1.0.0
	 */
	public function admin_enqueue_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		$suffix    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Register admin scripts.
		wp_register_script( 'everest-forms-save-and-continue-builder', plugins_url( "/assets/js/admin/admin{$suffix}.js", EVF_SAVE_AND_CONTINUE_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm', 'jquery-ui-sortable', 'perfect-scrollbar' ), EVF_SAVE_AND_CONTINUE_VERSION, true );
		wp_localize_script(
			'everest-forms-save-and-continue-builder',
			'everest_forms_save_and_continue_params',
			array(
				'ajax_url'                => admin_url( 'admin-ajax.php', 'relative' ),
				'ajax_nonce'              => wp_create_nonce( 'ajax_post_submission_nonce' ),
				'form_id'                 => isset( $_GET['form_id'] ) ? wp_unslash( $_GET['form_id'] ) : '', // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification
				'sc_disable_message'      => esc_html__( 'Save and Continue is currently disabled. Please enable it to activate this feature with Everest Forms.', 'everest-forms-save-and-continue' ),
				'sc_offline_form_disaled' => esc_html__( 'Offline form is disabled.', 'everest-forms-save-and-continue' ),
				'sc_disabled'             => esc_html__( 'Save and Continue Email can not be deleted !', 'everest-forms-save-and-continue' ),
				'sc_email_subject'        => esc_html__( 'Continue filling out this form later - {form_name} ', 'everest-forms-save-and-continue' ),
				'sc_email_message'        => esc_html__( 'Your form has been saved successfully, please visit {save_and_continue_url}  within {save_and_continue_time} to complete the form.<br><br>Thank You.', 'everest-forms-save-and-continue' ),
			)
		);

		// Admin scripts for EVF builder page.
		if ( 'everest-forms_page_evf-builder' === $screen_id ) {
			wp_enqueue_script( 'everest-forms-save-and-continue-builder' );
		}
	}

	/**
	 * Save and Continue Form Settings.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		new SaveAndContinue();
		// Instantiate Setting.
		new Setting();
		// Save Emtry.
		new SaveEntry();
		// Display Field.
		new DisplayFields();
	}

	/**
	 * Plugin Updater.
	 *
	 * @since 1.0.0
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return \EVF_Plugin_Updater::updates( EVF_SAVE_AND_CONTINUE_PLUGIN_FILE, 148284, EVF_SAVE_AND_CONTINUE_VERSION );
		}
	}

	/**
	 * Load Localization files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-save-and-continue/everest-forms-save-and-continue-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-save-and-continue-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-save-and-continue' );

		load_textdomain( 'everest-forms-save-and-continue', WP_LANG_DIR . '/everest-forms-save-and-continue/everest-forms-save-and-continue-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-save-and-continue', false, plugin_basename( dirname( EVF_SAVE_AND_CONTINUE_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @since 1.0.0
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EVF_SAVE_AND_CONTINUE_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-add-ons/save-and-continue/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms Save and Continue documentation', 'everest-forms-save-and-continue' ) . '">' . esc_html__( 'Docs', 'everest-forms-save-and-continue' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}

	/**
	 * Everest Forms fallback notice.
	 *
	 * @since 1.0.0
	 */
	public function everest_forms_missing_notice() {
		/* translators: %s: everest-forms version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Save and Continue requires at least %s or later to work!', 'everest-forms-save-and-continue' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms 1.7.7', 'everest-forms-save-and-continue' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Everest Forms Pro fallback notice.
	 *
	 * @since 1.0.0
	 */
	public function everest_forms_pro_missing_notice() {
		/* translators: %s: everest-forms-pro version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Save and Continue depends on the last version of %s or later to work!', 'everest-forms-save-and-continue' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms Pro 1.4.2', 'everest-forms-save-and-continue' ) . '</a>' ) . '</p></div>';
	}
}
