<?php
/**
 * Google Drive Integration.
 *
 * @package EverestForms\Pro\Integrations
 * @since   1.3.7
 */

namespace EverestForms\Pro\Integrations;

defined( 'ABSPATH' ) || exit;

/**
 * Google Drive Integration.
 */
class GoogleDrive extends \EVF_Integration {

	/**
	 * Client.
	 *
	 * @var object
	 */
	public $client;

	/**
	 * Init and hook in the integration.
	 */
	public function __construct() {
		$this->id                 = 'google_drive';
		$this->icon               = plugins_url( '/assets/img/google-drive.png', EFP_PLUGIN_FILE );
		$this->method_title       = esc_html__( 'Google Drive', 'everest-forms-pro' );
		$this->method_description = esc_html__( 'Google Drive Integration with Everest Forms', 'everest-forms-pro' );

		// Load the settings.
		$this->init_settings();
		$this->init_form_fields();

		// OAuth 2.0 credentials.
		$this->client_id     = 'OTgzMTQ3ODg2ODk2LXR0azhqdW8xZTE1NzB1bWtyN2lxczlpMnZqYWdidTY2LmFwcHMuZ29vZ2xldXNlcmNvbnRlbnQuY29t';
		$this->client_secret = 'WXVKcnVIOWVDRXNrU3lmRW5oM2NJQnJf';

		// Define user set variables.
		$this->auth_code      = $this->get_option( 'auth_code' );
		$this->access_token   = $this->get_option( 'access_token' );
		$this->refresh_token  = $this->get_option( 'refresh_token' );
		$this->integration    = $this->get_integration();
		$this->account_status = $this->is_auth_required() ? 'disconnected' : 'connected';

		// Get Google API client.
		if ( $this->is_integration_page() ) {
			$this->client = $this->get_client();
		}

		// Google Client API Authentication.
		add_action( 'everest_forms_integration_account_connect_' . $this->id, array( $this, 'api_authenticate' ) );
		add_action( 'everest_forms_integration_account_disconnect_' . $this->id, array( $this, 'api_deauthenticate' ) );
	}

	/**
	 * Initialize integration settings form fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'auth_code' => array(
				'title'   => __( 'Enter Google Access Code', 'everest-forms-pro' ),
				'type'    => 'text',
				'default' => '',
			),
		);
	}

	/**
	 * Is authentication required?
	 *
	 * @return bool
	 */
	public function is_auth_required() {
		return empty( $this->access_token ) || empty( $this->refresh_token );
	}

	/**
	 * Returns an authorized API client.
	 *
	 * @return \Google_Client the authorized client object
	 *
	 * @link https://developers.google.com/sheets/api/quickstart/php
	 *
	 * @param bool $is_ajax If called from Ajax.
	 */
	public function get_client( $is_ajax = false ) {
		$logger = evf_get_logger();

		// Load client only once.
		if ( ! empty( $this->client ) && ! $is_ajax ) {
			return $this->client;
		}

		$client = new \Google_Client();
		$client->setClientId( base64_decode( $this->client_id ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
		$client->setClientSecret( base64_decode( $this->client_secret ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
		$client->setRedirectUri( 'urn:ietf:wg:oauth:2.0:oob' );
		$client->setApplicationName( 'Everest Forms - Google Drive API v' . EFP_VERSION );
		$client->setScopes( \Google_Service_Drive::DRIVE );
		$client->setAccessType( 'offline' );
		$client->setPrompt( 'select_account consent' );

		// Apply custom options to the client.
		$client = apply_filters( 'everest_forms_google_sheets_auth_get_client_custom_options', $client );

		// For debugging only.
		if (
			function_exists( 'wp_get_environment_type' )
			&& 'local' === wp_get_environment_type()
		) {
			$httpClient = new \GuzzleHttp\Client(
				array(
					'exceptions' => false,
					'verify'     => false, // Otherwise HTTPS requests will fail.
					'base_uri'   => $client->getConfig( 'base_path' ),
				)
			);
			$client->setHttpClient( $httpClient );
		}

		if ( $is_ajax ) {
			return $client;
		}

		if (
			! empty( $this->auth_code )
			&& $this->is_auth_required()
		) {
			try {
				// Exchange authorization code for an access token.
				$accessToken = $client->fetchAccessTokenWithAuthCode( $this->auth_code );
			} catch ( \Exception $e ) {
				$accessToken['error'] = $e->getMessage();
				$logger->error(
					sprintf( 'Unable to fetch access token with auth code: %s', $accessToken['error'] ),
					array(
						'source' => 'google-sheets',
					)
				);
			}

			// Bail if we have an error.
			if ( ! empty( $accessToken['error'] ) ) {
				return $client;
			}

			// Update access and refresh token.
			$this->update_option( 'access_token', $client->getAccessToken() );
			$this->update_option( 'refresh_token', $client->getRefreshToken() );
		}

		// Set the access token used for requests.
		if ( ! empty( $this->access_token ) ) {
			$client->setAccessToken( $this->access_token );
		}

		// Refresh the token if it's expired.
		if ( $client->isAccessTokenExpired() ) {
			$refresh = $client->getRefreshToken();
			if ( empty( $refresh ) && isset( $this->refresh_token ) ) {
				$refresh = $this->refresh_token;
			}

			if ( ! empty( $refresh ) ) {
				try {
					// Refresh the token if possible, else fetch a new one.
					$refreshToken = $client->fetchAccessTokenWithRefreshToken( $refresh );
				} catch ( \Exception $e ) {
					$refreshToken['error'] = $e->getMessage();
					$logger->error(
						sprintf( 'Unable to fetch access token with refresh token: %s', $refreshToken['error'] ),
						array(
							'source' => 'google-sheets',
						)
					);
				}

				// Bail if we have an error.
				if ( ! empty( $refreshToken['error'] ) ) {
					return $client;
				}

				// Update access and refresh token.
				$this->update_option( 'access_token', $client->getAccessToken() );
				$this->update_option( 'refresh_token', $client->getRefreshToken() );
			}
		}

		return $client;
	}

	/**
	 * Google Drive API authenticate.
	 *
	 * @param array $posted_data Posted client credentials.
	 */
	public function api_authenticate( $posted_data ) {
		$auth_code = $this->get_field_value( 'auth_code', $this->form_fields['auth_code'], $posted_data );

		// Is valid auth to proceed?
		if ( empty( $auth_code ) ) {
			wp_send_json_error(
				array(
					'error'     => esc_html__( 'Could not authenticate to the Google Drive.', 'everest-forms-pro' ),
					'error_msg' => esc_html__( 'Please provide the correct Google access code.', 'everest-forms-pro' ),
				)
			);
		}

		$client      = $this->get_client( true );
		$accessToken = $client->fetchAccessTokenWithAuthCode( $auth_code );

		if ( isset( $accessToken['access_token'] ) ) {
			$this->update_option( 'auth_code', $auth_code );
			$this->update_option( 'access_token', $client->getAccessToken() );
			$this->update_option( 'refresh_token', $client->getRefreshToken() );

			wp_send_json_success(
				array(
					'button'      => esc_html__( 'Remove Authentication', 'everest-forms-pro' ),
					'description' => esc_html__( 'Google Drive account authenticated.', 'everest-forms-pro' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'error'     => esc_html__( 'Could not authenticate to the Google Drive.', 'everest-forms-pro' ),
					'error_msg' => esc_html( $accessToken['error_description'] ),
				)
			);
		}
	}

	/**
	 * Google Drive API authenticate.
	 *
	 * @param array $posted_data Posted client credentials.
	 */
	public function api_deauthenticate( $posted_data ) {
		$this->init_settings();

		$client = $this->get_client( true );

		if (
			empty( $posted_data['key'] )
			&& $this->id === $posted_data['source']
		) {
			update_option( $this->get_option_key(), array() );
			wp_send_json_success(
				array(
					'remove'      => false,
					'oauth'       => filter_var( $client->createAuthUrl(), FILTER_SANITIZE_URL ),
					'button'      => esc_html__( 'Authenticate with Google account', 'everest-forms-pro' ),
					'description' => esc_html__( 'Get the Google access code, we will use that code to authenticate with Google.', 'everest-forms-pro' ),
				)
			);
		}
	}

	/**
	 * Facilitates mailchimp integration. Evoked from extensibility.
	 */
	public function output_integration() {
		?>
		<div class="everest-forms-integration-content">
			<div class="integration-addon-detail">
				<div class="evf-integration-info-header">
					<figure class="evf-integration-logo">
						<img src="<?php echo esc_attr( $this->icon ); ?>" alt="<?php echo esc_attr( 'Google Drive Icon' ); ?>">
					</figure>
					<div class="integration-info">
						<h3><?php echo $this->method_title; /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */ ?></h3>
						<div class="integration-status <?php echo esc_attr( $this->account_status ); ?>">
							<span class="toggle-switch <?php echo esc_attr( $this->account_status ); ?>">
								<?php if ( 'connected' === $this->account_status ) : ?>
									<?php esc_html_e( 'Connected', 'everest-forms-pro' ); ?>
								<?php endif; ?>
							</span>
						</div>
					</div>
					</div>
				<p><?php echo $this->method_description; /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */ ?></p>
			</div>
			<div class="integration-connection-detail">
				<div class="evf-account-connect">
					<h3><?php esc_attr_e( 'Authenticate Google Drive', 'everest-forms-pro' ); ?></h3>
					<?php if ( empty( $this->auth_code ) && $this->is_auth_required() ) : ?>
						<p><?php esc_html_e( 'Get the Google access code, we will use that code to authenticate with Google.', 'everest-forms-pro' ); ?></p>
					<?php else : ?>
						<p><?php esc_html_e( 'Google Drive account authenticated.', 'everest-forms-pro' ); ?></p>
					<?php endif; ?>
					<form>
						<div class="evf-connection-form hidden">
							<input type="text" name="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" id="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" class="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" placeholder="<?php esc_attr_e( 'Enter Google Access Code', 'everest-forms-pro' ); ?>" value="<?php echo esc_attr( $this->auth_code ); ?>">
							<button type="submit" class="everest-forms-btn everest-forms-btn-primary everest-forms-integration-connect-account" style="padding:7px 14px" data-source="<?php echo esc_attr( $this->id ); ?>"><?php esc_html_e( 'Verify access code', 'everest-forms-pro' ); ?></button>
						</div>
						<?php if ( empty( $this->auth_code ) && $this->is_auth_required() ) : ?>
							<a href="<?php echo esc_url( filter_var( $this->client->createAuthUrl(), FILTER_SANITIZE_URL ) ); ?>" class="everest-forms-btn everest-forms-btn-primary everest-forms-integration-open-window" data-source="<?php echo esc_attr( $this->id ); ?>">
								<?php esc_html_e( 'Authenticate with Google account', 'everest-forms-pro' ); ?>
							</a>
						<?php else : ?>
							<a href="#" class="everest-forms-btn everest-forms-btn-secondary everest-forms-integration-disconnect-account" data-source="<?php echo esc_attr( $this->id ); ?>">
								<?php esc_html_e( 'Remove Authentication', 'everest-forms-pro' ); ?>
							</a>
						<?php endif; ?>
					</form>
				</div>
			</div>
		</div>
		<?php
	}
}
