<?php
/**
 * EverestForms Pro setup
 *
 * @package EverestForms_Pro
 * @since   1.3.5
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main EverestForms Pro Class.
 *
 * @class EverestForms_Pro
 */
final class EverestForms_Pro {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		// Load plugin text domain.
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		// Checks with Everest Forms is installed.
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.7.0', '>=' ) ) {
			$this->define_constants();
			$this->includes();

			// Hooks.
			add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
			add_filter( 'everest_forms_get_builder_pages', array( $this, 'load_builder_pages' ) );
			add_filter( 'everest_forms_get_settings_pages', array( $this, 'load_settings_pages' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ), 11 );
			add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
			add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
			add_action( 'everest_forms_shortcode_scripts', array( $this, 'shortcode_scripts' ) );
			add_filter( 'everest_forms_builder_strings', array( $this, 'form_builder_strings' ) );
			add_filter( 'everest_forms_get_script_data', array( $this, 'form_script_data' ), 10, 2 );
			add_filter( 'everest_forms_get_settings_validation', array( $this, 'validation_settings' ) );
			add_filter( 'everest_forms_general_settings', array( $this, 'general_settings' ) );
			add_action( 'everest_forms_inline_email_settings', array( $this, 'apply_email_attacment_setting' ), 10, 2 );

			add_filter( 'everest_forms_get_field_settings_email', array( $this, 'field_settings_email' ) );
			add_action( 'everest_forms_email_validation', array( $this, 'email_validation' ), 10, 3 );

			// AutoPopulate.
			add_filter( 'everest_forms_get_field_settings_email', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_first-name', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_last-name', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_checkbox', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_radio', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_text', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_number', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_date-time', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_select', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_textarea', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_get_field_settings_url', array( $this, 'field_settings_autopopulate' ) );
			add_filter( 'everest_forms_field_properties', array( $this, 'evf_auto_populate_form_field' ), 99, 3 );

			// Entry actions.
			add_action( 'admin_init', array( $this, 'process_actions' ) );
			add_action( 'everest_forms_view_entries_notices', array( $this, 'entry_notices' ) );
			add_filter( 'everest_forms_entries_list_actions', array( $this, 'entries_list_actions' ) );
			add_action( 'everest_forms_custom_menu_count', array( $this, 'unread_menu_count' ) );
			add_filter( 'everest_forms_entry_table_actions', array( $this, 'entry_table_actions' ), 10, 2 );
			add_action( 'everest_forms_entry_details_sidebar_action', array( $this, 'display_action_button' ), 10, 2 );

			// Corn Job.
			add_action( 'everest_forms_cleanup_old_entries', array( $this, 'cleanup_old_entries' ) );

			add_action( 'before_delete_post', array( $this, 'delete_entry_files_before_form_delete' ), 10, 1 );
			add_action( 'everest_forms_before_delete_entries', array( $this, 'delete_entry_files' ), 10, 1 );

			// Entry processing and setup.
			add_filter( 'everest_forms_entry_statuses', array( $this, 'entry_statuses' ) );
			add_filter( 'everest_forms_entry_bulk_actions', array( $this, 'entry_bulk_actions' ) );
			add_filter( 'everest_forms_entries_table_columns', array( $this, 'entries_table_columns' ), 10, 2 );
			add_action( 'everest_forms_entry_table_column_value', array( $this, 'entries_table_column_value' ), 10, 3 );
			add_action( 'everest_forms_after_entry_details_hndle', array( $this, 'add_starred_icon' ) );
			add_action( 'everest_forms_after_entry_details', array( $this, 'entry_details_actions' ), 10, 3 );
			add_filter( 'everest_forms_hidden_entry_fields', array( $this, 'entry_hidden_fields' ), 20 );
			add_action( 'everest_forms_after_entry_details', array( $this, 'payment_details_inside_entry' ), 10, 2 );
			add_action( 'everest_forms_process_validate_payment-single', array( $this, 'payment_single_validation' ), 10, 4 );
			add_action( 'everest_forms_process_validate_payment-multiple', array( $this, 'payment_multiple_validation' ), 10, 4 );

			// Row meta.
			add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );

			// AJAX events.
			add_action( 'wp_ajax_everest_forms_entry_star', array( $this, 'ajax_entry_star' ) );
			add_action( 'wp_ajax_everest_forms_entry_read', array( $this, 'ajax_entry_read' ) );
			add_action( 'wp_ajax_everest_forms_update_entry', array( $this, 'ajax_update_entry' ) );
			add_action( 'wp_ajax_everest_forms_get_columns', array( $this, 'get_columns' ) );
			add_action( 'wp_ajax_everest_forms_set_columns', array( $this, 'set_columns' ) );
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_missing_notice' ) );
		}
	}


	/**
	 * Email Field Setting
	 *
	 * @param mixed $settings Settings.
	 *
	 * @return $settings Settings.
	 */
	public function field_settings_email( $settings ) {
		$options   = $settings['basic-options']['field_options'];
		$new_array = array();
		foreach ( $options as $option ) {
			$new_array [] = $option;
			if (
				'required' === $option && ! in_array( 'no_duplicates', $new_array )
			) {
				$new_array [] = 'no_duplicates';
			}
		}
		$settings['basic-options']['field_options'] = $new_array;
		return $settings;
	}

	/**
	 * Email Validation.
	 *
	 * @param string $field_id Field ID.
	 * @param mixed  $field_submit Field Submit.
	 * @param mixed  $form_data Form Data.
	 * @return void
	 */
	public function email_validation( $field_id, $field_submit, $form_data ) {
		// Duplicates Check.
		if ( ! empty( $form_data['form_fields'][ $field_id ]['no_duplicates'] ) ) {
			$value     = sanitize_email( $field_submit['primary'] );
			$duplicate = evf_search_entries(
				array(
					'form_id' => $form_data['id'],
					'search'  => $value,
				)
			);
			if ( ! empty( $duplicate ) ) {
				$invalid_email = esc_html__( 'Email already exists.', 'everest-forms-pro' );
				if ( empty( $form_data['form_fields'][ $field_id ]['confirmation'] ) ) {
					evf()->task->errors[ $form_data['id'] ][ $field_id ] = $invalid_email;
				} else {
					evf()->task->errors[ $form_data['id'] ][ $field_id ]['primary'] = $invalid_email;
				}
				update_option( 'evf_validation_error', 'yes' );

			}
		}
	}

	/**
	 *  Autopopulate Field Setting
	 *
	 * @param mixed $settings Settings.
	 *
	 * @return $settings Settings.
	 */
	public function field_settings_autopopulate( $settings ) {
		$options   = $settings['advanced-options']['field_options'];
		$new_array = array();
		foreach ( $options as $option ) {
			$new_array [] = $option;
			if (
				'css' === $option && ! in_array( 'enable_prepopulate', $new_array ) && ! in_array( 'parameter_name', $new_array )
			) {
				$new_array [] = array_push( $new_array, 'enable_prepopulate', 'parameter_name' );
			}
		}
		$settings['advanced-options']['field_options'] = $new_array;
		return $settings;
	}

	/**
	 * Auto Populate Form Field.
	 *
	 * @param string $properties Value.
	 * @param mixed  $field Field.
	 * @param mixed  $form_data Form Data.
	 * @return $properties Properties.
	 */
	public function evf_auto_populate_form_field( $properties, $field, $form_data ) {
		$get_url =	$get_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]"; //phpcs:ignore
		$get_url = parse_url( $get_url );
		if ( ! empty( $get_url['query'] ) ) {
			parse_str( $get_url['query'], $query_params );
			foreach ( $query_params as $key => $url ) {
				// populating form field value with query string paramter value.
				if ( isset( $field['enable_prepopulate'] ) && '1' === $field['enable_prepopulate'] ) {
					$param_name = $field['parameter_name'];
					if ( $param_name === $key ) {
						if ( 'checkbox' === $field['type'] || 'radio' === $field['type'] || 'select' === $field['type'] ) {
							foreach ( $field['choices'] as $key => $option_value ) {
								$selected  = ! empty( $option_value['default'] ) ? $option_value['default'] : '';
								$multi_val = explode( ',', $url );
								foreach ( $multi_val as $value ) {
									if ( $value === $option_value['label'] ) {
										$selected                                = 1;
										$properties['inputs'][ $key ]['default'] = $selected;
									}
								}
							}
						} else {
							$properties['inputs']['primary']['attr']['value'] = sanitize_text_field( $url );
						}
					}
				}
			}
		}
		return $properties;
	}


	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		// If the single instance hasn't been set, set it now.
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-pro/everest-forms-pro-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-pro-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-pro' );

		load_textdomain( 'everest-forms-pro', WP_LANG_DIR . '/everest-forms-pro/everest-forms-pro-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-pro', false, plugin_basename( dirname( EFP_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Includes.
	 */
	private function includes() {
		/**
		 * Abstract classes.
		 */
		include_once EFP_ABSPATH . 'includes/abstracts/class-evf-form-integration.php';
		include_once EFP_ABSPATH . 'includes/abstracts/class-evf-email-marketing.php';
		include_once EFP_ABSPATH . 'includes/abstracts/class-evf-form-fields-upload.php';
		include_once EFP_ABSPATH . 'includes/abstracts/class-evf-payments.php';

		/**
		 * Core classes.
		 */
		include_once EFP_ABSPATH . 'includes/payments/functions.php';
		include_once EFP_ABSPATH . 'includes/class-evf-conditional-logics.php';
		include_once EFP_ABSPATH . 'includes/class-evf-dashboard-widget.php';
		include_once EFP_ABSPATH . 'includes/class-evf-required-indicators.php';

		/**
		 * Additional fields.
		 */
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-html.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-title.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-divider.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-phone.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-hidden.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-address.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-country.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-password.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-file-upload.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-image-upload.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-signature.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-rating.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-privacy-policy.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-range-slider.php';

		/**
		 * Payment fields.
		 */
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-payment-single.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-payment-radio.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-payment-checkbox.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-payment-quantity.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-payment-total.php';
		include_once EFP_ABSPATH . 'includes/fields/class-evf-field-credit-card.php';

		/**
		 * Analytics.
		 */
		include_once EFP_ABSPATH . 'includes/analytics/class-everest-forms-analytics.php';
	}

	/**
	 * Plugin Updater.
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return EVF_Plugin_Updater::updates( EFP_PLUGIN_FILE, 3441, EFP_VERSION );
		}
	}

	/**
	 * Define EVF Constants.
	 */
	private function define_constants() {
		$this->define( 'EFP_ABSPATH', dirname( EFP_PLUGIN_FILE ) . '/' );
		$this->define( 'EFP_PLUGIN_BASENAME', plugin_basename( EFP_PLUGIN_FILE ) );
	}

	/**
	 * Define constant if not already set.
	 *
	 * @param string      $name  Constant name.
	 * @param string|bool $value Constant value.
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Load builder page.
	 *
	 * @param  array $builder Builder page.
	 * @return array of builder page.
	 */
	public function load_builder_pages( $builder ) {
		$builder[] = include_once EFP_ABSPATH . 'includes/builder/class-evf-builder-integrations.php';
		$builder[] = include_once EFP_ABSPATH . 'includes/builder/class-evf-builder-payments.php';

		return $builder;
	}

	/**
	 * Load settings page.
	 *
	 * @param  array $settings Settings page.
	 * @return array of settings page.
	 */
	public function load_settings_pages( $settings ) {
		$settings[] = include_once EFP_ABSPATH . 'includes/settings/class-evf-setting-payment.php';

		return $settings;
	}

	/**
	 * Admin Enqueue scripts.
	 */
	public function admin_enqueue_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		$suffix    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Range Slider Scripts.
		wp_register_style( 'ion-range-slider', plugins_url( '/assets/css/rangeSlider.css', EFP_PLUGIN_FILE ), array(), '2.3.1' );
		wp_register_script( 'ion-range-slider', plugins_url( '/assets/js/ion-range-slider/ion.rangeSlider' . $suffix . '.js', EFP_PLUGIN_FILE ), array( 'jquery' ), '2.3.1', true );

		// Register admin scripts.
		wp_register_style( 'everest-forms-pro-admin', plugins_url( '/assets/css/admin.css', EFP_PLUGIN_FILE ), array(), EFP_VERSION );
		wp_register_script( 'everest-forms-entries-scripts', plugins_url( "/assets/js/admin/everest-forms-pro-entries{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'updates' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-builder-scripts', plugins_url( "/assets/js/admin/everest-forms-pro-builder{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'ion-range-slider' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-integrations-scripts', plugins_url( "/assets/js/admin/integration{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-general-scripts', plugins_url( "/assets/js/admin/general{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-conditionals-scripts', plugins_url( "/assets/js/admin/conditional-logic{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-entry-dashboard-scripts', plugins_url( "/assets/js/admin/dashboard-analytics{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm' ), EFP_VERSION, true );

		// Add RTL support for admin styles.
		wp_style_add_data( 'everest-forms-pro-admin', 'rtl', 'replace' );

		// Admin styles for EVF pages only.
		if ( in_array( $screen_id, evf_get_screen_ids(), true ) ) {
			wp_enqueue_style( 'everest-forms-pro-admin' );
			wp_localize_script(
				'everest-forms-payment-scripts',
				'evfpayment_payment_params',
				array(
					'i18n_payment_option_label' => __( 'Payment Options', 'everest-forms-pro' ),
					'i18n_only_paypal_gateway'  => __( 'Paypal is selected as payment gateway.', 'everest-forms-pro' ),
					'i18n_only_stripe_gateway'  => __( 'Stripe is selected as payment gateway.', 'everest-forms-pro' ),
					'i18n_empty_gateways'       => __( 'Please enable payment gateways.', 'everest-forms-pro' ),
				)
			);
			// Localize scripts for entries table for sortable modal box.
			wp_localize_script(
				'everest-forms-admin',
				'evf_entries_params',
				array(
					'i18n_adjust_entries_columns_title' => esc_html__( 'Select Entries Table Column', 'everest-forms-pro' ),
					'i18n_adjust_entries_columns_description' => esc_html__( 'Drag & Drop fields to visible columns to show the particular field column in the entries table.', 'everest-forms-pro' ),
					'i18n_entries_save'                 => esc_html__( 'Save', 'everest-forms-pro' ),
					'i18n_entries_cancel'               => esc_html__( 'Cancel', 'everest-forms-pro' ),
					'i18n_entries_active_column_name'   => esc_html__( 'Visible Columns(Change)', 'everest-forms-pro' ),
					'i18n_entries_inactive_column_name' => esc_html__( 'Hidden Columns(Change)', 'everest-forms-pro' ),
					'ajax_entries_nonce'                => wp_create_nonce( 'process-entries-ajax-nonce' ),
					'ajax_url'                          => admin_url( 'admin-ajax.php', 'relative' ),
				)
			);

			wp_enqueue_script( 'everest-forms-conditionals-scripts' );
			wp_enqueue_script( 'everest-forms-integrations-scripts' );
			wp_enqueue_script( 'everest-forms-general-scripts' );

			if ( isset( $_GET['page'] ) && 'evf-entries' === $_GET['page'] && ! isset( $_GET['view-entry'] ) && ! isset( $_GET['edit-entry'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification
				$form_id   = isset( $_GET['form_id'] ) ? absint( $_GET['form_id'] ) : -1; //phpcs:ignore WordPress.Security.NonceVerification
				$form_obj  = EVF()->form->get( $form_id );
				$form_data = ! empty( $form_obj->post_content ) ? evf_decode( $form_obj->post_content ) : '';

				if ( ! isset( $form_data['settings']['enable_entries_dashboard_analytics'] ) || '1' === $form_data['settings']['enable_entries_dashboard_analytics'] ) {
					// Flatpickr.
					wp_enqueue_style( 'flatpickr' );
					wp_enqueue_script( 'flatpickr' );

					// Jquery block UI.
					wp_enqueue_style( 'jquery-blockui' );
					wp_enqueue_script( 'jquery-blockui' );

					// Jquery date picker.
					wp_enqueue_script( 'jquery-ui-datepicker' );

					// Random color.
					wp_enqueue_script( 'randomcolor', plugins_url( '/assets/js/randomcolor/randomColor' . $suffix . '.js', EFP_PLUGIN_FILE ), array(), '0.5.4', true );

					// Moment.
					wp_enqueue_script( 'moment', plugins_url( '/assets/js/moment/moment-with-locales.min.js', EFP_PLUGIN_FILE ), array(), '2.27.0', true );

					// Chartjs.
					wp_enqueue_style( 'chartjs', plugins_url( '/assets/css/Chart.min.css', EFP_PLUGIN_FILE ), array(), '2.3.9' );
					wp_enqueue_script( 'chartjs', plugins_url( '/assets/js/chartjs/Chart.bundle.min.js', EFP_PLUGIN_FILE ), array(), '2.3.9', true );

					wp_enqueue_script( 'everest-forms-entry-dashboard-scripts' );
				}
			}

			wp_localize_script(
				'everest-forms-conditionals-scripts',
				'evf_conditional_rules',
				array(
					'i18n_remove_rule'         => esc_html__( 'Remove existing rules?', 'everest-forms-pro' ),
					'i18n_remove_rule_message' => esc_html__( 'Payment rule is mutually exclusive. Do you wish to remove other rules?', 'everest-forms-pro' ),
				)
			);

			wp_localize_script(
				'everest-forms-integrations-scripts',
				'evfp_params',
				array(
					'admin_url'           => admin_url(),
					'ajax_url'            => admin_url( 'admin-ajax.php', 'relative' ),
					'i18n_ok'             => esc_html__( 'OK', 'everest-forms-pro' ),
					'i18n_close'          => esc_html__( 'Close', 'everest-forms-pro' ),
					'i18n_cancel'         => esc_html__( 'Cancel', 'everest-forms-pro' ),
					'ajax_nonce'          => wp_create_nonce( 'process-ajax-nonce' ),
					'form_id'             => isset( $_GET['form_id'] ) ? wp_unslash( $_GET['form_id'] ) : '', // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				'i18n_confirm_save'       => esc_html__( 'We need to save your progress to continue to the Marketing panel. Is that OK?', 'everest-forms-pro' ),
				'i18n_confirm_connection' => esc_html__( 'Are you sure you want to delete this connection?', 'everest-forms-pro' ),
				'i18n_prompt_connection'  => esc_html__( 'Enter a %type% nickname', 'everest-forms-pro' ),
				'i18n_prompt_placeholder' => esc_html__( 'Eg: Newsletter %type%', 'everest-forms-pro' ),
				'i18n_error_name'         => esc_html__( 'You must provide a %type% nickname', 'everest-forms-pro' ),
				'i18n_required_field'     => esc_html__( 'Field required', 'everest-forms-pro' ),
				'provider_auth_error'     => esc_html__( 'Could not authenticate with the provider.', 'everest-forms-pro' ),
				'required_field'          => esc_html__( 'Fields are required.', 'everest-forms-pro' ),
				)
			);

			if ( 'everest-forms_page_evf-entries' === $screen_id ) {
				wp_enqueue_script( 'everest-forms-entries-scripts' );
				wp_localize_script(
					'everest-forms-entries-scripts',
					'everest_forms_entries',
					array(
						'nonce'                   => wp_create_nonce( 'everest-forms-entry' ),
						'ajax_url'                => admin_url( 'admin-ajax.php', 'relative' ),
						'entry_star'              => esc_html__( 'Star entry', 'everest-forms-pro' ),
						'entry_unstar'            => esc_html__( 'Unstar entry', 'everest-forms-pro' ),
						'entry_read'              => esc_html__( 'Mark entry read', 'everest-forms-pro' ),
						'entry_unread'            => esc_html__( 'Mark entry unread', 'everest-forms-pro' ),
						'entry_update_nonce'      => wp_create_nonce( 'everest_forms_entry_update' ),
						'unload_confirmation_msg' => esc_html__( 'The changes you made will be lost if you navigate away from this page.', 'everest-forms-pro' ),
					)
				);

				// Load only on edit entry page.
				if ( isset( $_GET['form_id'], $_GET['edit-entry'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
						include_once EVF_ABSPATH . 'includes/class-evf-frontend-scripts.php';

						$atts = array(
							'id' => (int) $_GET['form_id'], // phpcs:ignore WordPress.Security.NonceVerification
						);

						// Load frontend scripts.
						$this->frontend_enqueue_scripts();
						EVF_Frontend_Scripts::load_scripts();

						// Load shortcode script.
						$this->shortcode_scripts( $atts, 'edit-entry' );

						// Load field scripts action.
						do_action( 'everest_forms_shortcode_scripts', $atts );
				}
			}

			// EverestForms builder pages.
			if ( in_array( $screen_id, array( 'everest-forms_page_evf-builder' ), true ) ) {
				wp_enqueue_style( 'ion-range-slider' );
				wp_enqueue_script( 'everest-forms-builder-scripts' );
				wp_localize_script(
					'everest-forms-builder-scripts',
					'everest_forms_builder',
					array(
						'i18n_field_rating_greater_than_max_value_error' => esc_html__( 'Please enter in a value less than 100.', 'everest-forms-pro' ),
						'i18n_privacy_policy_consent_message' => esc_html__( 'I allow this website to collect and store the submitted data.', 'everest-forms-pro' ),
						'i18n_required' => esc_html__( '(Required)', 'everest-forms-pro' ),
					)
				);
			}
		}
	}

	/**
	 * Load shortcode scripts.
	 *
	 * @param array  $atts Shortcode attributes.
	 * @param string $page Conditionally load script on certain pages.
	 */
	public function shortcode_scripts( $atts, $page = '' ) {
		$form_data = evf()->form->get( $atts['id'], array( 'content_only' => true ) );

		if ( ! empty( $form_data['form_fields'] ) ) {
			$is_phone = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type'         => 'phone',
					'phone_format' => 'smart',
				)
			);

			$is_country = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type'                => 'country',
					'enable_country_flag' => 1,
				)
			);

			$is_address = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type'                => 'address',
					'enable_country_flag' => 1,
				)
			);

			$is_password = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type'              => 'password',
					'password_strength' => 1,
				)
			);

			$validate_password = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type'                => 'password',
					'password_validation' => 1,
				)
			);

			$is_file_upload = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type' => 'file-upload',
				)
			);

			$is_image_upload = wp_list_filter(
				$form_data['form_fields'],
				array(
					'type' => 'image-upload',
				)
			);

			if ( ! empty( $is_phone ) || ! empty( $is_country ) || ! empty( $is_address ) ) {
				wp_enqueue_style( 'jquery-intl-tel-input' );

				if ( ! empty( $is_phone ) ) {
					wp_enqueue_script( 'jquery-intl-tel-input' );
				}

				if ( ! empty( $is_country ) || ! empty( $is_address ) ) {
					wp_enqueue_style( 'select2' );
					wp_enqueue_script( 'selectWoo' );
				}
			}

			if ( ! empty( $is_password ) ) {
				wp_enqueue_script( 'evf-password-strength-meter' );
			}

			if ( ! empty( $validate_password ) ) {
				wp_enqueue_script( 'evf-password-validation' );
			}

			if ( ! empty( $is_file_upload ) || ! empty( $is_image_upload ) ) {
				wp_enqueue_script( 'everest-forms-file-upload' );
			}

			// Load scripts for edit entry.
			if ( 'edit-entry' === $page ) {
				$is_email = wp_list_filter(
					$form_data['form_fields'],
					array(
						'type' => 'email',
					)
				);

				$is_datetime = wp_list_filter(
					$form_data['form_fields'],
					array(
						'type' => 'date-time',
					)
				);

				if ( ! empty( $is_datetime ) ) {
					wp_enqueue_style( 'flatpickr' );
					wp_enqueue_script( 'flatpickr' );
				}

				if ( ! empty( $is_email ) && (bool) apply_filters( 'everest_forms_mailcheck_enabled', true ) ) {
					wp_enqueue_script( 'mailcheck' );
				}

				// Load frontend scripts.
				wp_enqueue_script( 'everest-forms-pro' );

				// Load editable entry scripts.
				do_action( 'everest_forms_editable_entry_scripts', $form_data );
			}
		}
	}

	/**
	 * Append additional strings for form builder.
	 *
	 * @since 1.3.0
	 *
	 * @param array $strings List of strings.
	 *
	 * @return array
	 */
	public function form_builder_strings( $strings ) {
		$currency   = get_option( 'everest_forms_currency', 'USD' );
		$currencies = evf_get_currencies();

		$strings['currency']            = sanitize_text_field( $currency );
		$strings['currency_name']       = sanitize_text_field( $currencies[ $currency ]['name'] );
		$strings['currency_decimal']    = sanitize_text_field( $currencies[ $currency ]['decimal_separator'] );
		$strings['currency_thousands']  = sanitize_text_field( $currencies[ $currency ]['thousands_separator'] );
		$strings['currency_symbol']     = sanitize_text_field( $currencies[ $currency ]['symbol'] );
		$strings['currency_symbol_pos'] = sanitize_text_field( $currencies[ $currency ]['symbol_pos'] );

		return $strings;
	}

	/**
	 * Frontend Enqueue scripts.
	 */
	public function frontend_enqueue_scripts() {
		$strings = array();
		$suffix  = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_enqueue_style( 'everest-forms-pro-frontend', plugins_url( '/assets/css/everest-forms-pro-frontend.css', EFP_PLUGIN_FILE ), array(), EFP_VERSION );

		wp_enqueue_script( 'everest-forms-pro', plugins_url( "/assets/js/frontend/everest-forms-pro{$suffix}.js", EFP_PLUGIN_FILE ), array( 'everest-forms' ), EFP_VERSION, true );
		wp_enqueue_script( 'conditional-logic-builder', plugins_url( "/assets/js/frontend/conditional-logic-frontend{$suffix}.js", EFP_PLUGIN_FILE ), array( 'everest-forms' ), EFP_VERSION, true );
		wp_localize_script(
			'everest-forms-pro',
			'everest_forms_pro_params',
			array(
				'plugin_url' => plugin_dir_url( EFP_PLUGIN_FILE ),
			)
		);

		// Range Slider Scripts.
		wp_register_style( 'ion-range-slider', plugins_url( '/assets/css/rangeSlider.css', EFP_PLUGIN_FILE ), array(), EFP_VERSION );
		wp_register_script( 'ion-range-slider', plugins_url( "/assets/js/ion-range-slider/ion.rangeSlider{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery' ), '2.3.1', true );

		// Signature pad scripts.
		wp_register_script( 'signature-pad', plugins_url( '/assets/js/signature_pad/signature_pad.umd.js', EFP_PLUGIN_FILE ), array( 'jquery' ), EFP_VERSION, true );
		wp_register_script( 'everest-forms-signature', plugins_url( "/assets/js/frontend/signature{$suffix}.js", EFP_PLUGIN_FILE ), array( 'everest-forms', 'signature-pad' ), EFP_VERSION, true );

		// Smart phone field scripts.
		wp_register_style( 'jquery-intl-tel-input', plugins_url( '/assets/css/intlTelInput.css', EFP_PLUGIN_FILE ), array(), EFP_VERSION );
		wp_register_script( 'jquery-intl-tel-input', plugins_url( "/assets/js/intlTelInput/jquery.intlTelInput{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery' ), '16.0.7', true );

		// Password strength meter scripts.
		wp_register_script( 'evf-password-strength-meter', plugins_url( "/assets/js/frontend/password-strength-meter{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery', 'password-strength-meter' ), EFP_VERSION, true );
		wp_register_script( 'evf-password-validation', plugins_url( "/assets/js/frontend/password-validation{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery' ), EFP_VERSION, true );
		wp_localize_script(
			'evf-password-validation',
			'everest_forms_password_params',
			array(
				'one_number'            => esc_html__( 'One number', 'everest-forms-pro' ),
				'one_lowercase'         => esc_html__( 'One lowercase character', 'everest-forms-pro' ),
				'one_uppercase'         => esc_html__( 'One uppercase character', 'everest-forms-pro' ),
				'one_special_character' => esc_html__( 'One special character', 'everest-forms-pro' ),
				'min_length'            => esc_html__( 'Minumum 8 characters', 'everest-forms-pro' ),
				'strong'                => esc_html__( 'Your password is strong', 'everest-forms-pro' ),
				'weak'                  => esc_html__( 'Your password is weak', 'everest-forms-pro' ),
			)
		);
		// File and image upload field scripts.
		wp_register_script( 'dropzone', plugins_url( "/assets/js/dropzone/dropzone{$suffix}.js", EFP_PLUGIN_FILE ), array( 'jquery' ), '5.5.0', true );
		wp_register_script( 'everest-forms-file-upload', plugins_url( "/assets/js/frontend/everest-forms-file-upload{$suffix}.js", EFP_PLUGIN_FILE ), array( 'dropzone', 'wp-util' ), EFP_VERSION, true );
		wp_localize_script(
			'everest-forms-file-upload',
			'everest_forms_upload_parms',
			array(
				'url'             => admin_url( 'admin-ajax.php' ),
				'errors'          => array(
					'file_not_uploaded' => esc_html__( 'This file was not uploaded.', 'everest-forms-pro' ),
					'file_limit'        => esc_html__( 'File limit has been reached ({fileLimit}).', 'everest-forms-pro' ),
					'file_extension'    => get_option( 'everest_forms_fileextension_validation' ),
					'file_size'         => get_option( 'everest_forms_filesize_validation', __( 'File exceeds max size allowed.', 'everest-forms-pro' ) ),
					'post_max_size'     => sprintf(
						/* translators: %s: Max upload size */
						esc_html__( 'File exceeds the upload limit allowed (%s).', 'everest-forms-pro' ),
						evf_max_upload()
					),
				),
				'loading_message' => esc_html__( 'Do not submit the form until the upload process is finished', 'everest-forms-pro' ),
			)
		);

		if ( function_exists( 'evf_get_currencies' ) ) {
			$currency                       = get_option( 'everest_forms_currency', 'USD' );
			$currencies                     = evf_get_currencies();
			$strings['currency_code']       = $currency;
			$strings['currency_thousands']  = $currencies[ $currency ]['thousands_separator'];
			$strings['currency_decimal']    = $currencies[ $currency ]['decimal_separator'];
			$strings['currency_symbol']     = $currencies[ $currency ]['symbol'];
			$strings['currency_symbol_pos'] = $currencies[ $currency ]['symbol_pos'];
		}
		$strings = apply_filters( 'everest_forms_frontend_strings', $strings );

		foreach ( (array) $strings as $key => $value ) {
			if ( ! is_scalar( $value ) ) {
				continue;
			}
			$strings[ $key ] = html_entity_decode( (string) $value, ENT_QUOTES, 'UTF-8' );
		}

		echo "<script type='text/javascript'>\n";
		echo "/* <![CDATA[ */\n";
		echo 'var evf_settings = ' . wp_json_encode( $strings ) . "\n";
		echo "/* ]]> */\n";
		echo "</script>\n";

		do_action( 'everest_forms_footer_end' );
	}

	/**
	 * Load Gutenberg block scripts.
	 */
	public function enqueue_block_editor_assets() {
		if ( has_block( 'everest-forms/form-selector' ) ) {
			wp_enqueue_style( 'everest-forms-pro-frontend', plugins_url( '/assets/css/everest-forms-pro-frontend.css', EFP_PLUGIN_FILE ), array(), EFP_VERSION );
		}
	}

	/**
	 * Add new settings to the validation settings page.
	 *
	 * @since  1.0.0
	 * @param  mixed $settings array of settings.
	 * @return mixed
	 */
	public function validation_settings( $settings ) {
		$new_settings = array(
			array(
				'title'    => __( 'Phone Number', 'everest-forms-pro' ),
				'desc'     => __( 'Enter the message for valid phone number.', 'everest-forms-pro' ),
				'default'  => __( 'Please enter a valid phone number.', 'everest-forms-pro' ),
				'css'      => 'min-width: 350px;',
				'id'       => 'everest_forms_phone_validation',
				'type'     => 'text',
				'desc_tip' => true,
			),
			array(
				'title'    => __( 'File Extension', 'everest-forms-pro' ),
				'desc'     => __( 'Enter the message for the allowed file extensions.', 'everest-forms-pro' ),
				'default'  => __( 'File type is not allowed.', 'everest-forms-pro' ),
				'css'      => 'min-width: 350px;',
				'id'       => 'everest_forms_fileextension_validation',
				'type'     => 'text',
				'desc_tip' => true,
			),
			array(
				'title'    => __( 'File Size', 'everest-forms-pro' ),
				'desc'     => __( 'Enter the message for the max file size allowed.', 'everest-forms-pro' ),
				'default'  => __( 'File exceeds max size allowed.', 'everest-forms-pro' ),
				'id'       => 'everest_forms_filesize_validation',
				'css'      => 'min-width: 350px;',
				'type'     => 'text',
				'desc_tip' => true,
			),
		);

		// Add new settings to the existing ones.
		foreach ( $settings as $key => $setting ) {
			if ( isset( $setting['id'] ) && 'everest_forms_number_validation' === $setting['id'] ) {
				array_splice( $settings, $key + 1, 0, $new_settings );
				break;
			}
		}

		return $settings;
	}

	/**
	 * Enable Auto delete entries.
	 *
	 * @param mixed $settings Settings.
	 */
	public function general_settings( $settings ) {
		array_splice(
			$settings,
			3,
			0,
			array(
				array(
					'title'   => esc_html__( 'Enable scheduled entry delete', 'everest-forms' ),
					'desc'    => esc_html__( 'Enable entry delete on schedule time.', 'everest-forms' ),
					'id'      => 'everest_forms_scheduled_entry_delete',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'title'   => esc_html__( 'Delete entries older than', 'everest-forms' ),
					'type'    => 'select',
					'id'      => 'everest_forms_scheduled_entry_delete_time',
					'default' => '30',
					'options' => array(
						'1'   => 'day',
						'7'   => 'week',
						'30'  => 'month',
						'90'  => 'quarter',
						'180' => 'half year',
						'365' => 'year',
					),
				),
			)
		);
		return $settings;
	}

	/**
	 * Add additional bulk actions.
	 *
	 * @param array $actions Bulk actions.
	 */
	public function entry_bulk_actions( $actions ) {
		return array_merge(
			array(
				'star'   => esc_html__( 'Star', 'everest-forms-pro' ),
				'unstar' => esc_html__( 'Unstar', 'everest-forms-pro' ),
				'read'   => esc_html__( 'Mark Read', 'everest-forms-pro' ),
				'unread' => esc_html__( 'Mark Unread', 'everest-forms-pro' ),
			),
			$actions
		);
	}

	/**
	 * Add additional entry statues.
	 *
	 * @param array $statuses Entry statuses.
	 */
	public function entry_statuses( $statuses ) {
		$position     = array_search( 'trash', array_keys( $statuses ), true );
		$new_statuses = array(
			'unread'  => esc_html__( 'Unread', 'everest-forms-pro' ),
			'starred' => esc_html__( 'Starred', 'everest-forms-pro' ),
		);

		return array_merge( array_slice( $statuses, 0, $position ), $new_statuses, array_slice( $statuses, $position ) );
	}

	/**
	 * Entries table column.
	 *
	 * @param array $columns Columns.
	 * @param array $form_data Forms data.
	 */
	public function entries_table_columns( $columns, $form_data ) {
		$new_columns    = array();
		$paypal_enabled = isset( $form_data['payments']['paypal']['enable_paypal'] ) ? '1' === $form_data['payments']['paypal']['enable_paypal'] : false;
		$stripe_enabled = isset( $form_data['payments']['stripe']['enable_stripe'] ) ? '1' === $form_data['payments']['stripe']['enable_stripe'] : false;

		if ( empty( $_GET['status'] ) || ( isset( $_GET['status'] ) && 'trash' !== $_GET['status'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$new_columns['indicators'] = '';
			$columns                   = array_merge( array_slice( $columns, 0, 1 ), $new_columns, array_slice( $columns, 1 ) );
		}

		if ( $paypal_enabled || $stripe_enabled ) {
			$new_columns['status'] = esc_html__( 'Status', 'everest-forms-pro' );
			$pos                   = array_search( 'date', array_keys( $columns ), true );
			$columns               = array_merge( array_slice( $columns, 0, $pos ), $new_columns, array_slice( $columns, $pos ) );
		}

		return $columns;
	}

	/**
	 * Renders the columns.
	 *
	 * @param  string $value Entry value.
	 * @param  object $entry Entry object.
	 * @param  string $column_name Column Name.
	 * @return string
	 */
	public function entries_table_column_value( $value, $entry, $column_name ) {
		if ( isset( $entry->meta ) ) {
			$payment_meta = $entry->meta;

			if ( isset( $payment_meta['meta'] ) ) {
				$payment_details = json_decode( $payment_meta['meta'] );
			}
		}

		switch ( $column_name ) {
			case 'total':
				if ( isset( $payment_details->payment_total ) ) {
					$amount = evf_sanitize_amount( $payment_details->payment_total, $payment_details->payment_currency );
					$total  = evf_format_amount( $amount, true, $payment_details->payment_currency );
					$value  = $total;
				} else {
					$value = '<span class="na">&mdash;</span>';
				}
				break;
			case 'status':
				if ( ! empty( $payment_meta['status'] ) ) {
					$dollar_icon = plugins_url( '/assets/img/icon-dollar.png', EFP_PLUGIN_FILE );
					$value       = ucfirst( $payment_meta['status'] ) . '<img src="' . $dollar_icon . '" alt="Payment">';
				} else {
					$value = '-';
				}
				break;
			case 'indicators':
				// Stars.
				$star_action = empty( $entry->starred ) ? 'star' : 'unstar';
				$star_title  = empty( $entry->starred ) ? esc_html__( 'Star entry', 'everest-forms-pro' ) : esc_html__( 'Unstar entry', 'everest-forms-pro' );
				$value       = '<a href="#" class="indicator-star ' . $star_action . '" data-id="' . absint( $entry->entry_id ) . '" title="' . esc_attr( $star_title ) . '"><span class="dashicons dashicons-star-filled"></span></a>';

				// Viewed.
				$read_action = empty( $entry->viewed ) ? 'read' : 'unread';
				$read_title  = empty( $entry->viewed ) ? esc_html__( 'Mark entry read', 'everest-forms-pro' ) : esc_html__( 'Mark entry unread', 'everest-forms-pro' );
				$value      .= '<a href="#" class="indicator-read ' . $read_action . '" data-id="' . absint( $entry->entry_id ) . '" title="' . esc_attr( $read_title ) . '"><span class="dashicons dashicons-marker"></span></a>';
				break;
		}

		return $value;
	}

	/**
	 * Ajax handler to toggle entry read state.
	 *
	 * @since 1.6.0
	 */
	public function ajax_entry_read() {
		$entry_id = isset( $_POST['entry_id'] ) ? absint( $_POST['entry_id'] ) : 0;

		check_ajax_referer( 'everest-forms-entry', 'nonce' );

		// Check for permissions.
		if ( ! current_user_can( 'everest_forms_view_entry', $entry_id ) ) {
			wp_send_json_error();
		}

		if ( ! empty( $_POST['task'] ) ) {
			$is_success = EVF_Admin_Entries::update_status( $entry_id, sanitize_key( $_POST['task'] ) );

			if ( $is_success ) {
				wp_send_json_success();
			}
		}

		wp_send_json_error();
	}

	/**
	 * Ajax handler to toggle entry stars.
	 *
	 * @since 1.6.0
	 */
	public function ajax_entry_star() {
		$entry_id = isset( $_POST['entry_id'] ) ? absint( $_POST['entry_id'] ) : 0;

		check_ajax_referer( 'everest-forms-entry', 'nonce' );

		// Check for permissions.
		if ( ! current_user_can( 'everest_forms_view_entry', $entry_id ) ) {
			wp_send_json_error();
		}

		if ( ! empty( $_POST['task'] ) ) {
			$is_success = EVF_Admin_Entries::update_status( $entry_id, sanitize_key( $_POST['task'] ) );

			if ( $is_success ) {
				wp_send_json_success();
			}
		}

		wp_send_json_error();
	}

	/**
	 * Ajax handler to update editable entry.
	 *
	 * @since 1.3.5
	 *
	 * @throws Exception If editable entry cannot be updated.
	 */
	public function ajax_update_entry() {
		global $wpdb;

		// Run a security check.
		check_ajax_referer( 'everest_forms_entry_update', 'security' );

		if ( ! isset( $_POST['data'] ) && ! current_user_can( 'manage_everest_forms' ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			wp_die( -1 );
		}

		try {
			if ( ! current_user_can( 'manage_everest_forms' ) ) {
				throw new Exception( esc_html__( 'Permissions is missing.', 'everest-forms-pro' ) );
			}

			parse_str( wp_unslash( $_POST['data'] ), $data ); // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated

			evf()->task->form_fields = array();

			// Get the form data.
			$form = evf()->form->get( (int) $data['everest_forms']['form_id'] );

			// Validate form is real and active (published).
			if ( ! $form || 'publish' !== $form->post_status ) {
				throw new Exception( esc_html__( 'Invalid form. Please check again.', 'everest-forms-pro' ) . ' ' . $data['everest_forms']['form_id'] );
			}

			// Formatted form data for hooks.
			$form_data = apply_filters( 'everest_forms_process_before_form_data', evf_decode( $form->post_content ), $data['everest_forms'] );

			// Pre-process filter.
			$entry = apply_filters( 'everest_forms_process_before_filter_entry_update', $data['everest_forms'], $form_data );

			// Validate editable entry is real.
			$entry_data = evf_get_entry( (int) $entry['entry_id'], true );
			if ( ! $entry_data->entry_id || $entry['entry_id'] !== $entry_data->entry_id ) {
				throw new Exception( esc_html__( 'Invalid entry ID. Please check again.', 'everest-forms-pro' ) . ' ' . $entry['entry_id'] );
			}

			// Validate form fields of editable form.
			$form_fields = isset( $entry['form_fields'] ) ? (array) $entry['form_fields'] : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( ! $form_fields ) {
				throw new Exception( esc_html__( 'Sorry! This entry does not have any fields.', 'everest-forms-pro' ) );
			}

			// Format fields.
			foreach ( $form_data['form_fields'] as $field ) {
				$field_id     = $field['id'];
				$field_key    = isset( $field['meta-key'] ) ? $field['meta-key'] : '';
				$field_type   = $field['type'];
				$field_submit = isset( $entry['form_fields'][ $field_id ] ) ? $entry['form_fields'][ $field_id ] : '';

				// Format only whitelisted editable entry fields.
				if ( ! apply_filters( 'everest_forms_admin_entries_edit_field_output_editable', $this->is_field_entries_editable( $field['type'] ), $field ) ) {
					continue;
				}

				// Ignore Fields inside repeater field.
				$repeater_fields = array_key_exists( 'repeater-fields', $field ) ? $field['repeater-fields'] : 'no';
				if ( 'no' !== $repeater_fields || 'repeater-fields' === $field['type'] ) {
					do_action( "everest_forms_process_format_{$field_type}", $field_id, $field_submit, $form_data, $field_key );
				}
			}

			// Post-process filter.
			$form_fields = apply_filters( 'everest_forms_process_filter_entry_update', evf()->task->form_fields, $entry, $form_data );

			// Post-process hook.
			do_action( 'everest_forms_process_form', $form_fields, $entry, $form_data );

			// Merge entry fields update with the latest updated field values.
			$entry_fields = array_merge( json_decode( $entry_data->fields, true ), $form_fields );

			// Update entry.
			$wpdb->update(
				$wpdb->prefix . 'evf_entries',
				array(
					'fields' => wp_json_encode( $entry_fields ),
				),
				array(
					'form_id'  => (int) $entry['form_id'],
					'entry_id' => (int) $entry['entry_id'],
				)
			);

			// Update form fields meta data.
			foreach ( $form_fields as $field ) {
				$field = apply_filters( 'everest_forms_entry_save_fields', $field, $form_data, $entry['entry_id'] );

				// Add only whitelisted editable fields to entry meta.
				if ( ! apply_filters( 'everest_forms_admin_entries_edit_field_output_editable', $this->is_field_entries_editable( $field['type'] ), $field ) ) {
					continue;
				}

				// Update entry meta.
				if ( isset( $field['meta_key'], $field['value'] ) ) {
					$wpdb->update(
						$wpdb->prefix . 'evf_entrymeta',
						array(
							'meta_value' => maybe_serialize( $field['value'] ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
						),
						array(
							'entry_id' => (int) $entry['entry_id'],
							'meta_key' => $field['meta_key'],
						)
					);
				}
			}

			// Removing Entries Cache.
			wp_cache_delete( $entry['entry_id'], 'evf-entry' );
			wp_cache_delete( $entry['entry_id'], 'evf-entrymeta' );
			wp_cache_delete( $entry['form_id'], 'evf-entries-ids' );
			wp_cache_delete( $entry['form_id'], 'evf-last-entries-count' );
			wp_cache_delete( $entry['form_id'], 'evf-search-entries' );
			wp_cache_delete( EVF_Cache_Helper::get_cache_prefix( 'entries' ) . '_unread_count', 'entries' );

			wp_send_json_success( array( 'message' => 'Entry updated successfully!' ) );
		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => $e->getMessage() ) );
		}

		wp_send_json_success();
	}

	/**
	 * Add starred icon if needed.
	 *
	 * @param object $entry Entry data.
	 */
	public function add_starred_icon( $entry ) {
		echo '1' === $entry->starred ? '<span class="dashicons dashicons-star-filled"></span>' : '';
	}

	/**
	 * Entry details action metabox.
	 *
	 * @param object $entry      Submitted entry values.
	 * @param object $entry_meta Entry meta data.
	 * @param array  $form_data  Form data and settings.
	 */
	public function entry_details_actions( $entry, $entry_meta, $form_data ) {
		$is_viewed    = false;
		$action_links = array();

		// Marked entry as read.
		if ( '1' !== $entry->viewed && empty( $_GET['action'] ) && empty( $_GET['unread'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$is_viewed = EVF_Admin_Entries::update_status( absint( $entry->entry_id ), 'read' );
		}

		$base_url = add_query_arg(
			array(
				'page'     => 'evf-entries',
				'form_id'  => absint( $form_data['id'] ),
				'entry_id' => absint( $entry->entry_id ),
			),
			admin_url( 'admin.php' )
		);

		$action_links['star'] = array(
			'url'   => wp_nonce_url( add_query_arg( array( 'action' => '1' === $entry->starred ? 'unstar' : 'star' ), $base_url ), 'starred-entry' ),
			'icon'  => '1' === $entry->starred ? 'dashicons-star-empty' : 'dashicons-star-filled',
			'label' => '1' === $entry->starred ? esc_html__( 'Unstar', 'everest-forms-pro' ) : esc_html__( 'Star', 'everest-forms-pro' ),
		);

		if ( '1' === $entry->viewed || $is_viewed ) {
			$action_links['read'] = array(
				'url'   => wp_nonce_url( add_query_arg( array( 'action' => 'unread' ), $base_url ), 'unread-entry' ),
				'icon'  => 'dashicons-hidden',
				'label' => esc_html__( 'Mark Unread', 'everest-forms-pro' ),
			);
		}

		$action_links['export'] = array(
			'url'   => wp_nonce_url( add_query_arg( array( 'action' => 'export_csv' ), $base_url ), 'export-entry' ),
			'icon'  => 'dashicons-download',
			'label' => esc_html__( 'Export Entry (CSV)', 'everest-forms-pro' ),
		);

		$quiz_enabled = isset( $form_data['settings']['enable_quiz'] ) && '1' === $form_data['settings']['enable_quiz'] ? true : false;

		if ( true === $quiz_enabled ) {
			$action_links['export-quiz-report-csv'] = array(
				'url'   => wp_nonce_url( add_query_arg( array( 'action' => 'export_quiz_report_csv' ), $base_url ), 'export-quiz-report-csv' ),
				'icon'  => 'dashicons-download',
				'label' => esc_html__( 'Export Quiz Report (CSV)', 'everest-forms-pro' ),
			);
		}

		if ( ! empty( $entry->fields ) ) {
			$action_links['notifications'] = array(
				'url'   => wp_nonce_url( add_query_arg( array( 'action' => 'notification' ), $base_url ), 'resend-entry' ),
				'icon'  => 'dashicons-email',
				'label' => esc_html__( 'Resend Notifications', 'everest-forms-pro' ),
			);
		}

		$action_links = apply_filters( 'everest_forms_entry_details_sidebar_actions_link', $action_links, $entry, $form_data );
		?>
		<!-- Entry Actions metabox -->
		<div id="everest-forms-entry-actions" class="stuffbox">
			<h2><?php esc_html_e( 'Entry Actions', 'everest-forms-pro' ); ?></h2>
			<div class="inside">
				<div class="everest-forms-entry-actions-meta">
			<?php
			foreach ( $action_links as $slug => $action_link ) {
				$target = ! empty( $action_link['target'] ) ? 'target="_blank" rel="noopener noreferrer"' : '';
				printf( '<p class="everest-forms-entry-%s">', esc_attr( $slug ) );
					printf( '<a href="%s" %s>', esc_url( $action_link['url'] ), $target ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						printf( '<span class="dashicons %s"></span>', esc_attr( $action_link['icon'] ) );
						echo esc_html( $action_link['label'] );
					echo '</a>';
				echo '</p>';
			}

			do_action( 'everest_forms_entry_details_sidebar_actions', $entry, $form_data );
			?>
				</div>
			</div>
		</div>
			<?php
	}

	/**
	 * Entries admin actions.
	 */
	public function process_actions() {
		$form_id = isset( $_GET['form_id'] ) ? absint( $_GET['form_id'] ) : '';

		if ( isset( $_GET['page'], $_GET['action'], $_GET['entry_id'] ) && 'evf-entries' === $_GET['page'] ) {
			$args     = array();
			$entry_id = absint( $_GET['entry_id'] );

			switch ( $_GET['action'] ) {
				case 'star':
				case 'unstar':
					check_admin_referer( 'starred-entry' );

					$starred = 'star' === $_GET['action'] ? 'starred' : 'unstarred';
					if ( EVF_Admin_Entries::update_status( $entry_id, sanitize_text_field( wp_unslash( $_GET['action'] ) ) ) ) {
						$args[ $starred ] = 1;
					}
					break;
				case 'unread':
					check_admin_referer( 'unread-entry' );

					if ( EVF_Admin_Entries::update_status( $entry_id, sanitize_text_field( wp_unslash( $_GET['action'] ) ) ) ) {
						$args['unread'] = 1;
					}
					break;
				case 'export_csv':
					check_admin_referer( 'export-entry' );

					$file_name = strtolower( get_the_title( $form_id . '-' . $entry_id ) );

					if ( $file_name ) {
						include_once EVF_ABSPATH . 'includes/export/class-evf-entry-csv-exporter.php';

						$exporter = new EVF_Entry_CSV_Exporter( $form_id, $entry_id );
						$exporter->set_filename( evf_get_csv_file_name( $file_name ) );
					}

					$exporter->export();
					break;

				case 'export_quiz_report_csv':
					check_admin_referer( 'export-quiz-report-csv' );

					$file_name = strtolower( get_the_title( $form_id . '-' . $entry_id ) );

					if ( $file_name ) {
						include_once EVF_ABSPATH . 'includes/export/class-evf-entry-csv-exporter.php';

						$exporter = new EVF_Entry_CSV_Exporter( $form_id, $entry_id );
						$exporter->set_filename( evf_get_csv_file_name( $file_name ) );
					}

					$exporter->export_quiz_report();
					break;

				case 'notification':
					check_admin_referer( 'resend-entry' );

					$entry = evf_get_entry( $entry_id );

					if ( ! empty( $entry->fields ) ) {
						$fields    = evf_decode( $entry->fields );
						$form_data = evf()->form->get( $form_id, array( 'content_only' => true ) );

						// Resend email notification.
						evf()->task->entry_email( $fields, array(), $form_data, $entry_id );
						$args['resend'] = 1;
					}
					break;
			}

			wp_safe_redirect(
				esc_url_raw(
					add_query_arg(
						array_merge(
							array(
								'form_id'    => $form_id,
								'view-entry' => $entry_id,
							),
							$args
						),
						admin_url( 'admin.php?page=evf-entries' )
					)
				)
			);
			exit();
		}
	}

	/**
	 * Entry action notices.
	 */
	public function entry_notices() {
		$message = '';

		if ( isset( $_GET['starred'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$message = esc_html__( 'This entry has been starred.', 'everest-forms-pro' );
		} elseif ( isset( $_GET['unstarred'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$message = esc_html__( 'This entry has been unstarred.', 'everest-forms-pro' );
		} elseif ( isset( $_GET['unread'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$message = esc_html__( 'This entry has been marked unread.', 'everest-forms-pro' );
		} elseif ( isset( $_GET['resend'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$message = esc_html__( 'Notifications were resent!', 'everest-forms-pro' );
		}

		if ( $message ) {
			echo '<div id="message" class="updated notice is-dismissible">';
			echo '<p>' . wp_kses_post( $message ) . '</p>';
			echo '</div>';
		}
	}

	/**
	 * Add unread entry total count to menu.
	 */
	public function unread_menu_count() {
		global $submenu;

		$cache_key = EVF_Cache_Helper::get_cache_prefix( 'entries' ) . '_unread_count';
		$count     = wp_cache_get( $cache_key, 'entries' );

		if ( false === $count ) {
			$forms = evf_get_all_forms();

			// Count unread entries from all forms.
			foreach ( $forms as $form_id => $form_title ) {
				$entries = evf_search_entries(
					array(
						'form_id' => $form_id,
						'status'  => 'unread',
						'limit'   => -1,
					)
				);

				if ( is_array( $entries ) ) {
					$count += count( $entries );
				}
			}

			wp_cache_add( $cache_key, $count, 'entries' );
		}

		// Add unread entry count.
		foreach ( $submenu['everest-forms'] as $key => $menu_item ) {
			if ( 0 === strpos( $menu_item[0], _x( 'Entries', 'Admin menu name', 'everest-forms-pro' ) ) ) {
				$submenu['everest-forms'][ $key ][0] .= ' <span class="awaiting-mod count-' . $count . '"><span class="unread-count">' . number_format_i18n( $count ) . '</span></span>'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
				break;
			}
		}
	}

	/**
	 * Check if the field entries are editable.
	 *
	 * @since 1.3.5
	 *
	 * @param string $field_type Field type.
	 *
	 * @return bool
	 */
	private function is_field_entries_editable( $field_type ) {
		return (bool) apply_filters(
			'everest_forms_entries_field_editable',
			in_array(
				$field_type,
				array(
					'first-name',
					'last-name',
					'text',
					'textarea',
					'select',
					'radio',
					'checkbox',
					'number',
					'email',
					'url',
					'date-time',
					'phone',
					'address',
					'country',
					'range-slider',
					'rating',
				),
				true
			),
			$field_type
		);
	}

	/**
	 * Edit Entry Form actions.
	 *
	 * @since 1.3.5
	 *
	 * @param bool $is_allowed True if allowed to execute. False if not.
	 */
	public function entries_list_actions( $is_allowed ) {
		if ( isset( $_GET['edit-entry'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			add_action( 'everest_forms_entries_list_actions_execute', array( $this, 'display_edit_page' ) );
			add_action( 'everest_forms_entry_details_edit_content', array( $this, 'display_edit_form' ), 10, 2 );
			return true;
		}

		return $is_allowed;
	}

	/**
	 * Edit Entry page.
	 *
	 * @since 1.3.5
	 */
	public function display_edit_page() {
		include 'views/html-admin-page-entry-edit.php';
	}

	/**
	 * Edit entry form metabox.
	 *
	 * @since 1.3.5
	 *
	 * @param object $entry     Submitted entry values.
	 * @param array  $form_data Form data and settings.
	 */
	public function display_edit_form( $entry, $form_data ) {
		$hide_empty   = isset( $_COOKIE['everest_forms_entry_hide_empty'] ) && 'true' === $_COOKIE['everest_forms_entry_hide_empty'];
		$entry_fields = apply_filters( 'everest_forms_entry_single_data', evf_decode( $entry->fields ), $entry, $form_data );
		?>
		<!-- Edit Entry Form metabox -->
		<div id="everest-forms-entry-fields" class="postbox">
			<div class="postbox-header">
				<h2 class="hndle">
				<?php do_action( 'everest_forms_before_edit_entry_details_hndle', $entry ); ?>
					<span>
				<?php
				/* translators: %s: Entry ID */
				printf( esc_html__( '%1$s: Entry #%2$s', 'everest-forms-pro' ), esc_html( _draft_or_post_title( $form_data['id'] ) ), absint( $entry->entry_id ) );
				?>
					</span>
				<?php do_action( 'everest_forms_after_edit_entry_details_hndle', $entry ); ?>
					<a href="#" class="everest-forms-empty-field-toggle">
					<?php $hide_empty ? esc_html_e( 'Show Empty Fields', 'everest-forms-pro' ) : esc_html_e( 'Hide Empty Fields', 'everest-forms-pro' ); ?>
					</a>
				</h2>
			</div>
			<div class="inside evf-field-container">
				<input type="hidden" name="everest_forms[form_id]" value="<?php echo esc_attr( $form_data['id'] ); ?>">
				<input type="hidden" name="everest_forms[entry_id]" value="<?php echo esc_attr( $entry->entry_id ); ?>">
				<table class="fixed evf-frontend-row wp-list-table widefat striped posts">
					<tbody class="evf-frontend-grid evf-grid-1">
					<?php
					if ( ! current_user_can( 'everest_forms_edit_entry', $entry->entry_id ) ) {
						echo '<p class="no-access">' . esc_html__( 'You do not have permission to edit this entry.', 'everest-forms-pro' ) . '</p>';
					} elseif ( empty( $entry_fields ) ) {
						// Whoops, no fields! This shouldn't happen under normal use cases.
						echo '<p class="no-fields">' . esc_html__( 'This entry does not have any fields', 'everest-forms-pro' ) . '</p>';
					} else {
						// Display the fields and their editable values.
						$this->display_edit_form_fields( $entry, $entry_fields, $form_data, $hide_empty );
					}
					?>
					</tbody>
				</table>
			</div>
		</div>
			<?php
	}

	/**
	 * Edit entry form fields.
	 *
	 * @since 1.3.5
	 *
	 * @param object $entry        Entry data.
	 * @param array  $entry_fields Entry fields data.
	 * @param array  $form_data    Form data and settings.
	 * @param bool   $hide_empty   Flag to hide empty fields.
	 */
	private function display_edit_form_fields( $entry, $entry_fields, $form_data, $hide_empty ) {
		// Display message if form fields doesn't exists.
		if ( empty( $form_data['form_fields'] ) || ! is_array( $form_data['form_fields'] ) ) {
			echo '<p class="everest-forms-entry-field-value">';
			if ( current_user_can( 'manage_everest_forms' ) ) {
				printf(
					wp_kses( /* translators: %s - form edit URL. */
						__( 'Seems like you don\'t have any fields in this form. <a href="%s">Add form fields!</a>', 'everest-forms-pro' ),
						array(
							'a' => array(
								'href' => array(),
							),
						)
					),
					esc_url( admin_url( 'admin.php?page=evf-builder&amp;tab=fields&amp;form_id=' . $entry->form_id ) )
				);
			} else {
				esc_html_e( 'Seems like you don\'t have any fields in this form.', 'everest-forms-pro' );
			}
			echo '</p>';
			return;
		}

		foreach ( $form_data['form_fields'] as $field_id => $field ) {
			$repeater_fields = array_key_exists( 'repeater-fields', $field ) ? $field['repeater-fields'] : 'no';
			if ( 'no' === $repeater_fields || 'repeater-fields' === $field['type'] ) {
				$this->display_edit_form_field( $field_id, $field, $entry_fields, $form_data, $hide_empty );
			}
		}
	}

	/**
	 * Edit entry form field.
	 *
	 * @since 1.3.5
	 *
	 * @param int   $field_id     Field id.
	 * @param array $field        Field data.
	 * @param array $entry_fields Entry fields data.
	 * @param array $form_data    Form data and settings.
	 * @param bool  $hide_empty   Flag to hide empty fields.
	 */
	private function display_edit_form_field( $field_id, $field, $entry_fields, $form_data, $hide_empty ) {
		// Check if the field can be displayed.
		if ( ! empty( $field['type'] ) && in_array( $field['type'], (array) apply_filters( 'everest_forms_entries_edit_fields_hidden', array( 'title', 'html' ) ), true ) ) {
			return;
		}

		$entry_field = ! empty( $entry_fields[ $field_id ] ) ? $entry_fields[ $field_id ] : $this->get_empty_entry_field_properties( $field );
		$field_value = ! empty( $entry_field['value'] ) ? $entry_field['value'] : '';
		$field_value = apply_filters( 'everest_forms_html_field_value', $field_value, $entry_field, $form_data, 'entry-single' );
		$field_class = ( is_string( $field_value ) && ( '(empty)' === wp_strip_all_tags( $field_value ) || '' === $field_value ) ) ? ' empty' : '';
		$field_style = $hide_empty && empty( $entry_field['value'] ) ? 'display:none;' : '';

		echo '<tr class="everest-forms-edit-entry-field field-name' . esc_attr( $field_class ) . '" style="' . esc_attr( $field_style ) . '"><th>';

		// Field label.
		printf(
			'<strong>%s</strong> %s',
			/* translators: %d - field ID. */
			! empty( $field['label'] ) ? esc_html( wp_strip_all_tags( $field['label'] ) ) : sprintf( esc_html__( 'Field ID #%d', 'everest-forms-pro' ), (int) $field_id ),
			! empty( $field['required'] ) ? apply_filters( 'everest_forms_field_required_label', '<abbr class="required" title="' . esc_attr__( 'Required', 'everest-forms-pro' ) . '">*</abbr>' ) : '' // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		);

		echo '</th></tr>';

		// Add properties to the field.
		$field['properties'] = EVF_Shortcode_Form::get_field_properties( $field, $form_data );

		// Field value.
		echo '<tr class="everest-forms-edit-entry-field field-value' . esc_attr( $field_class ) . '" style="' . esc_attr( $field_style ) . '"><td>';

		// Display entry editable and non-editable form field.
		if ( apply_filters( 'everest_forms_admin_entries_edit_field_output_editable', $this->is_field_entries_editable( $field['type'] ), $field ) ) {
			EVF_Shortcode_Form::wrapper_start( $field, $form_data );
			do_action( "everest_forms_display_edit_form_field_{$field['type']}", $entry_field, $field, $form_data );
		} else {
			if ( isset( $field_value['label'] ) ) {
				$field_value = is_array( $field_value['label'] ) ? implode( '<br />', $field_value['label'] ) : $field_value['label'];
			}

			echo ( is_string( $field_value ) && ( '(empty)' === wp_strip_all_tags( $field_value ) || '' === $field_value ) ) ? esc_html__( 'Empty', 'everest-forms-pro' ) : nl2br( make_clickable( $field_value ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		echo '</td></tr>';
	}

	/**
	 * Get empty entry field properties.
	 *
	 * @since 1.3.5
	 *
	 * @param array $properties Field properties.
	 *
	 * @return array Empty entry field properties.
	 */
	public function get_empty_entry_field_properties( $properties ) {
		return array(
			'name'      => ! empty( $properties['label'] ) ? $properties['label'] : '',
			'value'     => '',
			'value_raw' => '',
			'id'        => ! empty( $properties['id'] ) ? $properties['id'] : '',
			'type'      => ! empty( $properties['type'] ) ? $properties['type'] : '',
		);
	}

	/**
	 * Entry table actions.
	 *
	 * @since 1.3.5
	 *
	 * @param  array  $actions Action links.
	 * @param  object $entry   Entry object.
	 * @return array Modified Actions array.
	 */
	public function entry_table_actions( $actions, $entry ) {
		if ( 'trash' !== $entry->status && apply_filters( 'everest_forms_entry_view_enable', false ) ) {
			$edit_action = array(
				'edit' => '<a href="' . esc_url( admin_url( 'admin.php?page=evf-entries&amp;form_id=' . $entry->form_id . '&amp;edit-entry=' . $entry->entry_id ) ) . '">' . esc_html__( 'Edit', 'everest-forms-pro' ) . '</a>',
			);

			// Check edit permission and add edit action to the existing ones.
			if ( current_user_can( 'everest_forms_edit_entry', $entry->entry_id ) ) {
				array_splice( $actions, -1, -1, $edit_action );
			}
		}

		return $actions;
	}

	/**
	 * Entry Details action button on the single entry view/edit page.
	 *
	 * @since 1.3.5
	 *
	 * @param object $entry    Entry object.
	 * @param array  $form_data Forms data and settings.
	 */
	public function display_action_button( $entry, $form_data ) {
		if ( ! isset( $form_data['id'], $entry->entry_id ) || ! current_user_can( 'everest_forms_edit_entry', $entry->entry_id ) ) {
			return;
		}

		// Edit Entry URL.
		if ( isset( $_GET['page'], $_GET['view-entry'] ) && 'evf-entries' === $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification
			printf(
				'<div id="publishing-action">
					<a href="%s" class="button button-primary button-large">%s</a>
				</div>',
				esc_url( admin_url( 'admin.php?page=evf-entries&amp;form_id=' . $entry->form_id . '&amp;edit-entry=' . $entry->entry_id ) ),
				esc_html__( 'Edit', 'everest-forms-pro' )
			);
		} else {
			printf(
				'<div id="publishing-action">
					<button class="button button-primary button-large everest-forms-submit-button" id="everest-forms-edit-entry-update">%s</button>
					<a href="%s" class="button button-secondary button-large">%s</a>
				</div>',
				esc_html__( 'Update', 'everest-forms-pro' ),
				esc_url( admin_url( 'admin.php?page=evf-entries&amp;form_id=' . $entry->form_id . '&amp;view-entry=' . $entry->entry_id ) ),
				esc_html__( 'Cancel', 'everest-forms-pro' )
			);
		}
	}


	/**
	 * Delete Attachment after removing Entry.
	 *
	 * @param int $entry_id Entry ID for which file should be removed.
	 */
	public function delete_entry_files( $entry_id ) {
		$get_entry = evf_get_entry( $entry_id, 'meta' );
		if ( ! empty( $get_entry->meta ) ) {
			foreach ( $get_entry->meta as $meta_key => $meta_value ) {
				if ( empty( $meta_value ) ) {
					continue;
				}
				if ( preg_match( '/signature_/', $meta_key ) ) {
					if ( file_exists( $meta_value ) ) {
						wp_delete_file( $meta_value );
					}
				} else {
					$files = explode( "\n", $meta_value );
					foreach ( $files as $file ) {
						$uploaded_file = ABSPATH . preg_replace( '/.*wp-content/', 'wp-content', wp_parse_url( $file, PHP_URL_PATH ) );
						if ( file_exists( $uploaded_file ) ) {
							wp_delete_file( $uploaded_file );
						}
					}
				}
			}
		}
	}

	/**
	 * Remove Files Attached to the Entry of the Form.
	 *
	 * @param int $form_id Form ID to get required form data and remove files.
	 */
	public function delete_entry_files_before_form_delete( $form_id ) {
		$entries = evf_get_entries_ids( $form_id );
		if ( ! empty( $entries ) ) {
			foreach ( $entries as $entry_id ) {
				$this->delete_entry_files( $entry_id );
			}
		}
	}

	/**
	 * Payment Details within Entry.
	 *
	 * @param object $entry     Entry Data.
	 * @param array  $form_data Form Data Object.
	 */
	public function payment_details_inside_entry( $entry, $form_data ) {
		$entry = evf_get_entry( $entry->entry_id );
		if ( empty( $entry->meta['type'] ) || 'payment' !== $entry->meta['type'] ) {
			return;
		}
		$entry_meta  = json_decode( $entry->meta['meta'], true );
		$status      = ! empty( $entry->status ) ? ucwords( sanitize_text_field( $entry->status ) ) : esc_html__( 'Unknown', 'everest-forms-pro' );
		$currency    = ! empty( $entry_meta['payment_currency'] ) ? $entry_meta['payment_currency'] : get_option( 'everest_forms_currency', 'USD' );
		$total       = isset( $entry_meta['payment_total'] ) ? evf_format_amount( evf_sanitize_amount( $entry_meta['payment_total'], $currency ), true, $currency ) : '-';
		$note        = ! empty( $entry_meta['payment_note'] ) ? esc_html( $entry_meta['payment_note'] ) : '';
		$gateway     = esc_html( apply_filters( 'evf_entry_details_payment_gateway', '-', $entry_meta, $entry, $form_data ) );
		$transaction = esc_html( apply_filters( 'evf_entry_details_payment_transaction', '-', $entry_meta, $entry, $form_data ) );
		$mode        = ! empty( $entry_meta['payment_mode'] ) && 'test' === $entry_meta['payment_mode'] ? 'test' : 'production';

		switch ( $entry_meta['payment_gateway'] ) {
			case 'paypal_standard':
				$gateway = esc_html__( 'PayPal Standard', 'everest-forms-pro' );
				if ( ! empty( $entry_meta['payment_transaction'] ) ) {
					$type        = 'production' === $mode ? '' : 'sandbox.';
					$transaction = sprintf( '<a href="https://www.%spaypal.com/webscr?cmd=_history-details-from-hub&id=%s" target="_blank">%s</a>', $type, $entry_meta['payment_transaction'], $entry_meta['payment_transaction'] );
				}
				break;
			case 'stripe':
				$gateway = esc_html__( 'Stripe', 'everest-forms-pro' );
				if ( ! empty( $entry_meta['payment_transaction'] ) ) {
					$transaction = sprintf( '<a href="https://dashboard.stripe.com/payments/%s" target="_blank" rel="noopener noreferrer">%s</a>', $entry_meta['payment_transaction'], $entry_meta['payment_transaction'] );
				}
				if ( ! empty( $entry_meta['payment_customer'] ) ) {
					$customer = sprintf( '<a href="https://dashboard.stripe.com/customers/%s" target="_blank" rel="noopener noreferrer">%s</a>', $entry_meta['payment_customer'], $entry_meta['payment_customer'] );
				}
				if ( ! empty( $entry_meta['payment_interval'] ) ) {
					$total .= ' <span style="font-weight:400; color:#999; display:inline-block;margin-left:4px;"><span class="dashicons dashicons-controls-repeat"></span> ' . $entry_meta['payment_interval'] . '</span>';
				}
				break;
		}

		?>
		<!-- Entry Payment details metabox -->
		<div id="everest-forms-entry-payment" class="stuffbox">
			<h2><?php esc_html_e( 'Payment Details', 'everest-forms-pro' ); ?></h2>
			<div class="inside">
				<div class="everest-forms-entry-payment-meta">
					<p class="everest-forms-entry-payment-status">
					<?php
					printf(
						/* translators: %s - entry payment status. */
						esc_html__( 'Status: %s', 'everest-forms-pro' ),
						'<strong>' . esc_html( $status ) . '</strong>'
					);
					?>
					</p>
					<p class="everest-forms-entry-payment-total">
						<?php
						printf(
						/* translators: %s - entry payment total. */
							esc_html__( 'Total: %s', 'everest-forms-pro' ),
							'<strong>' . wp_kses_post( $total ) . '</strong>'
						);
						?>
					</p>
					<p class="everest-forms-entry-payment-gateway">
						<?php
						printf(
						/* translators: %s - entry payment gateway. */
							esc_html__( 'Gateway: %s', 'everest-forms-pro' ),
							'<strong>' . esc_html( $gateway ) . '</strong>'
						);
						if ( 'test' === $mode ) {
							printf( ' (%s)', esc_html( _x( 'Test', 'Gateway mode', 'everest-forms-pro' ) ) );
						}
						?>
					</p>
					<p class="everest-forms-entry-payment-transaction">
						<?php
						printf(
						/* translators: %s - entry payment transaction. */
							esc_html__( 'Transaction ID: %s', 'everest-forms-pro' ),
							'<strong>' . wp_kses_post( $transaction ) . '</strong>'
						);
						?>
					</p>
						<?php if ( ! empty( $customer ) ) : ?>
					<p class="everest-forms-entry-payment-customer">
							<?php
							printf(
							/* translators: %s - entry payment customer. */
								esc_html__( 'Customer ID: %s', 'everest-forms-pro' ),
								'<strong>' . $customer . '</strong>' // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
							);
							?>
					</p>
					<?php endif; ?>
				</div>
			</div>
		</div>
			<?php
	}

	/**
	 * Hide Payment entry field.
	 *
	 * @param  array $entry_fields Entry fields.
	 * @return array               List for fields.
	 */
	public function entry_hidden_fields( $entry_fields ) {
		return array_merge( $entry_fields, array( 'status', 'type', 'meta' ) );
	}

	/**
	 * Single Payment validation.
	 *
	 * @param string $field_id     Field ID.
	 * @param string $field_submit Field's submitted value.
	 * @param array  $form_data    Form data object.
	 * @param array  $field_type   Type of the field.
	 */
	public function payment_single_validation( $field_id, $field_submit, $form_data, $field_type ) {
		if ( isset( $form_data['form_fields'][ $field_id ]['required'] ) && empty( $field_submit ) && '0' !== $field_submit && 'user' === $form_data['form_fields'][ $field_id ]['item_type'] ) {
			$validation_text = get_option( 'evf_' . $field_type . '_validation', __( 'Please enter the desire amount.', 'everest-forms-pro' ) );
		}

		if ( isset( $validation_text ) ) {
			EVF()->task->errors[ $form_data['id'] ][ $field_id ] = apply_filters( 'everest_forms_type_validation', $validation_text );
			update_option( 'evf_validation_error', 'yes' );
		}
	}

	/**
	 * Multiple Payment validation.
	 *
	 * @param string $field_id     Field ID.
	 * @param string $field_submit Field submit flag.
	 * @param array  $form_data    Form data object.
	 * @param array  $field_type   Type of the field.
	 */
	public function payment_multiple_validation( $field_id, $field_submit, $form_data, $field_type ) {
		if ( isset( $form_data['fields'][ $field_id ]['required'] ) && empty( $field_submit ) && '0' !== $field_submit ) {
			$validation_text = get_option( 'evf_' . $field_type . '_validation', __( 'Please choose one Item.', 'everest-forms-pro' ) );
		}
		if ( isset( $validation_text ) ) {
			EVF()->task->errors[ $form_data['id'] ][ $field_id ] = apply_filters( 'everest_forms_type_validation', $validation_text );
			update_option( 'evf_validation_error', 'yes' );
		}
	}

	/**
	 * Form script data.
	 *
	 * @param  array  $params Array of l10n data parameters.
	 * @param  string $handle Script handle the data will be attached to.
	 * @return array
	 */
	public function form_script_data( $params, $handle ) {
		if ( 'everest-forms' === $handle ) {
			$params = array_merge(
				$params,
				array(
					'i18n_no_countries'           => _x( 'No countries found', 'enhanced select', 'everest-forms-pro' ),
					'i18n_messages_phone'         => get_option( 'everest_forms_phone_validation', __( 'Please enter a valid phone number.', 'everest-forms-pro' ) ),
					'i18n_messages_fileextension' => get_option( 'everest_forms_fileextension_validation', __( 'File type is not allowed.', 'everest-forms-pro' ) ),
					'i18n_messages_filesize'      => get_option( 'everest_forms_filesize_validation', __( 'File exceeds max size allowed.', 'everest-forms-pro' ) ),
				)
			);
		}

		return $params;
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EFP_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-pro/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms Pro documentation', 'everest-forms-pro' ) . '">' . esc_html__( 'Docs', 'everest-forms-pro' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}


	/**
	 * Form setting for admin and user.
	 *
	 * @param array   $setting       Connection setting for email attahcment.
	 * @param integer $connection_id Connection id for the attachment.
	 */
	public function apply_email_attacment_setting( $setting, $connection_id ) {
		$form_id   = isset( $_GET['form_id'] ) ? absint( $_GET['form_id'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification
		$form_obj  = EVF()->form->get( $form_id );
		$form_data = ( ! empty( $form_obj->post_content ) ) ? evf_decode( $form_obj->post_content ) : '';
		everest_forms_panel_field(
			'checkbox',
			'settings',
			'file-email-attachments',
			$form_data,
			esc_html__( 'Send File As Attachment', 'everest-forms-pro' )
		);
	}

	/**
	 * Everest Forms fallback notice.
	 */
	public function everest_forms_missing_notice() {
		/* translators: %s: everest-forms version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms Pro requires at least %s or later to work!', 'everest-forms-pro' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms 1.6.0', 'everest-forms-pro' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Scheduled Task.
	 *
	 * @since 1.4.5
	 */
	public function cleanup_old_entries() {
		if ( 'yes' !== get_option( 'everest_forms_scheduled_entry_delete' ) ) {
			return;
		}

		$forms = evf()->form->get_multiple( array(), true );

		foreach ( $forms as $form ) {
			$expires = get_option( 'everest_forms_scheduled_entry_delete_time', 30 );
			$entries = evf_search_entries(
				array(
					'form_id' => $form['id'],
				)
			);

			foreach ( $entries as $key => $entry ) {
				$entry = evf_get_entry( $entry );
				if ( strtotime( $entry->date_created . ' + ' . $expires . ' days' ) < time() ) {
					EVF_Admin_Entries::remove_entry( $entry->entry_id );
				}
			}
		}
	}

	 /**
	  * Sortable column submission form ajax.
	  *
	  * @since 1.4.4
	  */
	public static function set_columns() {
		try {
			if ( ! isset( $_POST['evf_entries_active_columns'] ) ) {
				wp_die( -1 );
			}
			// Check nonce security of form after submission.
			check_ajax_referer( 'process-entries-ajax-nonce', 'security' );

			// Get form_id, form_data and column names.
			$form_id   = isset( $_POST['evf_entries_form_id'] ) ? absint( $_POST['evf_entries_form_id'] ) : 0;
			$form_data = evf()->form->get( absint( $form_id ), array( 'content_only' => true ) );

			// Get the column value to update in the field.
			$active_column_value = str_replace( 'evf_field_', '', array_flip( wp_unslash( $_POST['evf_entries_active_columns'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$active_column_value = array_keys( array_flip( $active_column_value ) );

			// Get post_content of the form for the column name updates.
			$result = 0;
			if ( $form_data ) {
				$form_data['meta']['entry_columns'] = $active_column_value;
				$result                             = evf()->form->update( $form_id, $form_data );
			}
			if ( $form_id === $result ) {
				wp_send_json_success(
					array(
						'message' => esc_html__( 'Columns has been updated Successfully.', 'everest-forms-pro' ),
					)
				);
			}
		} catch ( Exception $e ) {
			wp_send_json_error(
				array(
					'message' => $e->getMessage(),
				)
			);
		}
	}

	/**
	 * Get entry columns.
	 *
	 * @since 1.4.4
	 */
	public static function get_columns() {
		try {
			// Check nonce security of form after submission.
			check_ajax_referer( 'process-entries-ajax-nonce', 'security' );
			// Get form_id, form_data and column names.
			$form_id   = isset( $_POST['evf_entries_form_id'] ) ? absint( $_POST['evf_entries_form_id'] ) : 0;
			$form_data = evf()->form->get( absint( $form_id ), array( 'content_only' => true ) );

			// Get the column list to show in the form.
			$columns = array();
			foreach ( $form_data['form_fields'] as $id => $field ) {
				if ( ! in_array( $field['type'], EVF_Admin_Entries_Table_List::get_columns_form_disallowed_fields(), true ) ) {
					$columns[ 'evf_field_' . $id ] = ! empty( $field['label'] ) ? wp_strip_all_tags( $field['label'] ) : esc_html__( 'Field', 'everest-forms-pro' );
				}
			}

			// Get all Columns.
			$all_columns = apply_filters( 'everest_forms_entries_table_form_fields_columns', $columns, $form_id, $form_data );
			// Get active Columns.
			$entries_table            = new EVF_Admin_Entries_Table_List();
			$entries_table->form_id   = $form_id;
			$entries_table->form_data = $form_data;
			$active_columns           = apply_filters( 'everest_forms_entries_table_form_fields_columns', $entries_table->get_columns_form_fields(), $form_id, $form_data );

			// Get inactive Columns.
			$inactive_columns = array_diff( $all_columns, $active_columns );

			wp_send_json(
				array(
					'inactive_columns'    => array_filter( $inactive_columns ),
					'active_columns'      => array_filter( $active_columns ),
					'evf_entries_form_id' => $form_id,
				)
			);
		} catch ( Exception $e ) {
			wp_send_json_error(
				array(
					'message' => $e->getMessage(),
				)
			);
		}
	}
}
